<?php

namespace Drupal\mercury_editor_page_templates\EventSubscriber;

use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;
use Symfony\Component\HttpFoundation\RedirectResponse;
use Drupal\Core\Url;

/**
 * Subscriber to intercept node add requests for content types with templates.
 */
class RequestSubscriber implements EventSubscriberInterface {

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    return [
      KernelEvents::REQUEST => ['onKernelRequest', 100],
    ];
  }

  /**
   * Intercepts requests to check for template redirection.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function onKernelRequest(RequestEvent $event) {
    $request = $event->getRequest();
    $path = $request->getPathInfo();

    // Check if this is a node/add request.
    if (preg_match('/^\/node\/add\/(.+)$/', $path, $matches)) {
      $content_type = $matches[1];

      // Check if this content type has templates configured.
      $templates = mercury_editor_page_templates_get_templates($content_type);
      if (!empty($templates)) {
        // Check if we're not already in the template selection process.
        if (!$request->query->has('template_applied')) {
          // Redirect to template selection for this content type.
          $redirect_url = Url::fromRoute('mercury_editor_page_templates.content_type_templates', [
            'node_type' => $content_type,
          ], [
            'query' => ['mercury_editor' => 1],
          ]);

          $response = new RedirectResponse($redirect_url->toString());
          $event->setResponse($response);
        }
      }
    }
  }

}
