<?php

namespace Drupal\mercury_editor_page_templates\TwigExtension;

use Twig\Extension\AbstractExtension;
use Twig\TwigFunction;

/**
 * Twig extension for layout label functionality.
 */
class LayoutLabelExtension extends AbstractExtension {

  /**
   * {@inheritdoc}
   */
  public function getFunctions() {
    return [
      new TwigFunction('get_layout_label', $this->getLayoutLabel(...)),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getName() {
    return 'mercury_editor_page_templates.layout_label_extension';
  }

  /**
   * Gets the human-readable label for a layout machine name.
   *
   * @param string $layout_id
   *   The layout machine name.
   *
   * @return string
   *   The layout label or the original layout_id if not found.
   */
  public function getLayoutLabel($layout_id) {
    if (empty($layout_id)) {
      return '';
    }

    try {
      $layout_definition = \Drupal::service('plugin.manager.core.layout')
        ->getDefinition($layout_id);

      if ($layout_definition) {
        return $layout_definition->getLabel();
      }
    }
    catch (\Exception $e) {
      // If we can't get the layout definition, fall back to the original ID.
      \Drupal::logger('mercury_editor_page_templates')->warning(
        'Could not get layout definition for @layout_id: @message',
        [
          '@layout_id' => $layout_id,
          '@message' => $e->getMessage(),
        ]
      );
    }

    // Fallback to a human-readable version of the machine name.
    return str_replace(['_', '-'], ' ', $layout_id);
  }

}
