<?php

namespace Drupal\mermaid_graphapi\Mermaid;

use Fhaculty\Graph\Edge\Directed;
use Fhaculty\Graph\Graph;
use Fhaculty\Graph\Vertex;

/**
 * The Mermaid flowchart format.
 */
class Flowchart {

  /**
   * Array of valid info for Mermaid Flowchart vertex shapes.
   */
  protected const VERTEX_SHAPES = [
    'box' => [
      'label' => 'Box',
      'open_string' => '[',
      'close_string' => ']',
    ],
    'rounded-box' => [
      'label' => 'Rounded box',
      'open_string' => '(',
      'close_string' => ')',
    ],
    'stadium' => [
      'label' => 'Stadium',
      'open_string' => '([',
      'close_string' => '])',
    ],
    'circle' => [
      'label' => 'Circle',
      'open_string' => '((',
      'close_string' => '))',
    ],
    'double-circle' => [
      'label' => 'Double circle',
      'open_string' => '(((',
      'close_string' => ')))',
    ],
    'subroutine' => [
      'label' => 'Subroutine',
      'open_string' => '[[',
      'close_string' => ']]',
    ],
    'database' => [
      'label' => 'Database',
      'open_string' => '[(',
      'close_string' => ')]',
    ],
    'asymmetric' => [
      'label' => 'Asymmetric',
      'open_string' => '>',
      'close_string' => ']',
    ],
    'rhombus' => [
      'label' => 'Rhombus',
      'open_string' => '{',
      'close_string' => '}',
    ],
    'hexagon' => [
      'label' => 'Hexagon',
      'open_string' => '{{',
      'close_string' => '}}',
    ],
    'parallelogram' => [
      'label' => 'Parallelogram',
      'open_string' => '[/',
      'close_string' => '/]',
    ],
    'parallelogram-alt' => [
      'label' => 'Parallelogram (alt)',
      'open_string' => '[\\',
      'close_string' => '\]',
    ],
    'trapezoid' => [
      'label' => 'Trapezoid',
      'open_string' => '[/',
      'close_string' => '\]',
    ],
    'trapezoid-alt' => [
      'label' => 'Trapezoid (alt)',
      'open_string' => '[\\',
      'close_string' => '/]',
    ],
  ];

  /**
   * Array of valid info for Mermaid Flowchart edge styles.
   */
  protected const EDGE_STYLES = [
    # N.B. Open lines are described in the Mermaid docs, but render as arrows.
    # See: https://mermaid.js.org/syntax/flowchart.html#an-open-link
    #'line_normal' => [
    #  'label' => 'Line (normal)',
    #  'style_string' => '---',
    #],
    #'line_thick' => [
    #  'label' => 'Line (thick)',
    #  'style_string' => '===',
    #],
    #'line_dotted' => [
    #  'label' => 'Line (dotted)',
    #  'style_string' => '-.-',
    #],
    'arrow_normal' => [
      'label' => 'Arrow (normal)',
      'style_string' => '-->',
    ],
    'arrow_thick' => [
      'label' => 'Arrow (thick)',
      'style_string' => '==>',
    ],
    'arrow_dotted' => [
      'label' => 'Arrow (dotted)',
      'style_string' => '-.->',
    ],
    'invisible' => [
      'label' => 'Invisible',
      'style_string' => '~~~',
    ],
    'circle' => [
      'label' => 'Circle',
      'style_string' => '--o',
    ],
    'cross' => [
      'label' => 'Cross',
      'style_string' => '--x',
    ],
    # @TODO: Add multi-directional arrows?
    # See: https://mermaid.js.org/syntax/flowchart.html#multi-directional-arrows
  ];

  /**
   * Array of valid info for Mermaid Flowchart graph directions.
   */
  protected const GRAPH_DIRECTIONS = [
    'left-right' => [
      'label' => 'Left to right',
      'direction_string' => 'LR',
    ],
    'right-left' => [
      'label' => 'Right to left',
      'direction_string' => 'RL',
    ],
    'top-bottom' => [
      'label' => 'Top to bottom',
      'direction_string' => 'TB',
    ],
    'bottom-top' => [
      'label' => 'Bottom to top',
      'direction_string' => 'BT',
    ],
  ];

  /**
   * The default vertex shape to use, if none is set as a vertex attribute.
   */
  protected $default_vertex_shape = '';

  /**
   * The default edge style to use, if none is set as an edge attribute.
   */
  protected $default_edge_style = '';

  /**
   * The default graph direction to use, if none is set as a graph attribute.
   */
  protected $default_graph_direction = '';

  /**
   * A basic constructor.
   */
  public function __construct(
    string $default_vertex_shape = '',
    string $default_edge_style = '',
    string $default_graph_direction = '',
  ) {
    if ($default_vertex_shape) {
      $this->default_vertex_shape = $default_vertex_shape;
    }
    if ($default_edge_style) {
      $this->default_edge_style = $default_edge_style;
    }
    if ($default_graph_direction) {
      $this->default_graph_direction = $default_graph_direction;
    }
  }

  /**
   * Return info for all Mermaid Flowchart vertex shapes.
   */
  public static function getVertexShapesInfo(): array {
    return self::VERTEX_SHAPES;
  }

  /**
   * Return info for all Mermaid Flowchart edge styles.
   */
  public static function getEdgeStylesInfo(): array {
    return self::EDGE_STYLES;
  }

  /**
   * Return info for all Mermaid Flowchart graph directions.
   */
  public static function getGraphDirectionsInfo(): array {
    return self::GRAPH_DIRECTIONS;
  }

  /**
   * Generate the Mermaid text to graph type.
   */
  public function getGraphText(Graph $graph): array {
    $mermaid_graph_text = [];

    $direction = $this->getGraphDirection($graph);
    $direction_string = $this->getGraphDirectionString($direction);
    $mermaid_graph_text[] = "flowchart {$direction_string}";

    return $mermaid_graph_text;
  }

  /**
   * Generate the Mermaid text to represent a vertex.
   */
  public function getVertexText(Vertex $vertex): array {
    $mermaid_graph_text = [];

    $id = $vertex->getId();
    $label = $vertex->getAttribute('title');
    $shape = $this->getVertexShape($vertex);
    $open_string = $this->getVertexOpenString($shape);
    $close_string = $this->getVertexCloseString($shape);
    $mermaid_graph_text[] = "    {$id}{$open_string}{$label}{$close_string}";

    array_push($mermaid_graph_text, ...$this->getVertexInteractionText($vertex));

    return $mermaid_graph_text;
  }

  /**
   * Generate the Mermaid text to represent a directed edge.
   *
   * @TODO: This plugin only supports directed edges at the moment.
   */
  public function getEdgeText(Directed $edge): array {
    $start_id = $edge->getVertexStart()->getId();
    $end_id = $edge->getVertexEnd()->getId();
    $style = $this->getEdgeStyle($edge);
    $style_string = $this->getEdgeStyleString($style);
    $label_string = $this->getEdgeLabelString($edge);
    return ["    {$start_id} {$style_string}{$label_string} {$end_id}"];
  }

  /**
   * Return a valid the graph direction name.
   */
  protected function getGraphDirection(Graph $graph): string {
    $direction = $graph->getAttribute('direction') ?: $this->default_graph_direction;
    if (!array_key_exists($direction, self::getGraphDirectionsInfo())) {
      $message = t("':direction' is not among the known Mermaid graph directions. Setting to default: :default_direction",
        [':direction' => $direction, ':default_direction' => $this->default_graph_direction]);
      \Drupal::logger('mermaid_graphapi')->error($message);
      $direction = $this->default_graph_direction;
    }
    return $direction;
  }

  /**
   * Return the string to represent the edge line.
   */
  protected function getGraphDirectionString(string $direction): string {
    $directions_info = self::getGraphDirectionsInfo();
    return $directions_info[$direction]['direction_string'];
  }

  /**
   * Return a valid the vertex shape name.
   */
  protected function getVertexShape(Vertex $vertex): string {
    $shape = $vertex->getAttribute('shape') ?: $this->default_vertex_shape;
    if (!array_key_exists($shape, self::getVertexShapesInfo())) {
      $message = t("':shape' is not among the known Mermaid vertex shapes. Setting to default: :default_shape",
        [':shape' => $shape, ':default_shape' => $this->default_vertex_shape]);
      \Drupal::logger('mermaid_graphapi')->error($message);
      $shape = $this->default_vertex_shape;
    }
    return $shape;
  }

  /**
   * Return the string to open the vertex shape.
   */
  protected function getVertexOpenString(string $shape): string {
    $shapes_info = self::getVertexShapesInfo();
    return $shapes_info[$shape]['open_string'];
  }

  /**
   * Return the string to close the vertex shape.
   */
  protected function getVertexCloseString(string $shape): string {
    $shapes_info = self::getVertexShapesInfo();
    return $shapes_info[$shape]['close_string'];
  }

  /**
   * Make the vertex a clickable link to its entity.
   */
  protected function getVertexInteractionText(Vertex $vertex): array {
    $mermaid_graph_text = [];
    $entity_path = $vertex->getAttribute('path');
    if ($entity_path) {
      $mermaid_graph_text[] = "    click " . $vertex->getId() . " href \"" . $entity_path . "\"";
    }
    return $mermaid_graph_text;
  }

  /**
   * Return a valid the edge style name.
   */
  protected function getEdgeStyle(Directed $edge): string {
    $style = $edge->getAttribute('style') ?: $this->default_edge_style;
    if (!array_key_exists($style, self::getEdgeStylesInfo())) {
      $message = t("':style' is not among the known Mermaid edge styles. Setting to default: :default_style",
        [':style' => $style, ':default_style' => $this->default_edge_style]);
      \Drupal::logger('mermaid_graphapi')->error($message);
      $style = $this->default_edge_style;
    }
    return $style;
  }

  /**
   * Return the string to represent the edge line.
   */
  protected function getEdgeStyleString(string $style): string {
    $styles_info = self::getEdgeStylesInfo();
    return $styles_info[$style]['style_string'];
  }

  /**
   * Return the string to represent the edge label.
   */
  protected function getEdgeLabelString(Directed $edge): string {
    $edge_label = $edge->getAttribute('title') ?: NULL;
    if ($edge_label) {
      return '| '. $edge_label . ' |';
    }
    return '';
  }

}
