<?php

namespace Drupal\mermaid_graphapi\Plugin\GraphFormat;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\graphapi\Plugin\GraphFormat\GraphFormatBase;
use Drupal\mermaid_graphapi\Mermaid\Flowchart;
use Fhaculty\Graph\Graph;

/**
 * The Mermaid flowchart graph format.
 *
 * Supports the following options in the render element '#options' array:
 *  - theme: The Mermaid theme.
 *
 * Supports the following attributes on graph vertices:
 *  - title: The label of the vertex.
 *  - path: The path to the entity represented by the vertex.
 *  - shape: The shape to represent the vertex.
 *
 * Supports the following attributes on graph edges:
 *  - style: The style to represent the edge.
 *
 * @GraphFormat(
 *   id = "mermaid_flowchart",
 *   label = @Translation("Mermaid flowchart"),
 *   engine = "mermaid",
 *   default_options = {
 *     "theme" = "default",
 *     "default_vertex_shape" = "circle",
 *     "default_edge_style" = "arrow_normal",
 *     "default_graph_direction" = "left-right",
 *   },
 *   supported_attributes = {
 *     "graph" = {
 *       "direction",
 *     },
 *     "vertex" = {
 *       "path",
 *       "shape",
 *     },
 *     "edge" = {
 *       "style",
 *       "title",
 *     },
 *   },
 * )
 */
class MermaidFlowchart extends GraphFormatBase {

  /**
   * Array of valid keys in the init directive.
   */
  protected const INIT_KEYS = [
    'theme',
  ];

  /**
   * {@inheritdoc}
   */
  public function defaultOptionsForm(array $form, FormStateInterface $form_state, EntityInterface $entity) {
    $form = parent::defaultOptionsForm($form, $form_state, $entity);

    // Clean up unused options. We can't unset() these, since the GraphAPI
    // submit handler hardcodes them.
    $form['height']['#access'] = FALSE;
    $form['width']['#access'] = FALSE;

    $form['theme'] = [
      '#type' => 'radios',
      '#title' => $this->t('Theme'),
      '#options' => [
        'default' => $this->t('Default'),
        'dark' => $this->t('Dark'),
        'forest' => $this->t('Forest'),
      ],
      '#default_value' => $entity->getOption('theme'),
    ];

    $shape_options = $style_options = $direction_options = [];

    foreach (Flowchart::getVertexShapesInfo() as $shape => $info) {
      $shape_options[$shape] = $this->t($info['label']);
    }
    $form['default_vertex_shape'] = [
      '#type' => 'select',
      '#title' => $this->t('Vertex shape'),
      '#options' => $shape_options,
      '#default_value' => $entity->getOption('default_vertex_shape'),
    ];

    foreach (Flowchart::getEdgeStylesInfo() as $style => $info) {
      $style_options[$style] = $this->t($info['label']);
    }
    $form['default_edge_style'] = [
      '#type' => 'select',
      '#title' => $this->t('Edge style'),
      '#options' => $style_options,
      '#default_value' => $entity->getOption('default_edge_style'),
    ];

    foreach (Flowchart::getGraphDirectionsInfo() as $direction => $info) {
      $direction_options[$direction] = $this->t($info['label']);
    }
    $form['default_graph_direction'] = [
      '#type' => 'select',
      '#title' => $this->t('Graph direction'),
      '#options' => $direction_options,
      '#default_value' => $entity->getOption('default_graph_direction'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function preProcess(&$variables) {
    $options = $variables['options'];
    $graph = $variables['graph'];
    $mermaid_graph_text = [];

    array_push($mermaid_graph_text, ...$this->processInitString($options));
    array_push($mermaid_graph_text, ...$this->processChartType($graph, $options));
    array_push($mermaid_graph_text, ...$this->processVertices($graph, $options));
    array_push($mermaid_graph_text, ...$this->processEdges($graph, $options));

    $variables['content'] = implode("\n", $mermaid_graph_text);
  }

  /**
   * Assemble init options if any are present.
   */
  protected function processInitString(array $options): array {
    $mermaid_graph_text = $init_options = $init_strings = [];
    foreach (static::INIT_KEYS as $init_key) {
      if (isset($options[$init_key])) {
        $init_options[$init_key] = $options[$init_key];
      }
    }
    array_walk($init_options, function ($value, $key) use (&$init_strings) {
      $init_strings[] = "'$key': '$value'";
    });
    if ($init_strings) {
      $mermaid_graph_text[] = "%%{init: { " . implode(', ', $init_strings) . " } }%%";
    }
    return $mermaid_graph_text;
  }

  /**
   * Add Mermaid chart type.
   */
  protected function processChartType(Graph $graph, array $options): array {
    $flowchart = new Flowchart(default_graph_direction: $options['default_graph_direction']);
    return $flowchart->getGraphText($graph);
  }

  /**
   * Process graph vertices to generate Mermaid text.
   */
  protected function processVertices(Graph $graph, array $options): array {
    $mermaid_graph_text = [];
    $flowchart = new Flowchart(default_vertex_shape: $options['default_vertex_shape']);
    foreach ($graph->getVertices() as $vertex) {
      array_push($mermaid_graph_text, ...$flowchart->getVertexText($vertex));
    }
    return $mermaid_graph_text;
  }

  /**
   * Process graph edges to generate Mermaid text.
   */
  protected function processEdges(Graph $graph, array $options): array {
    $mermaid_graph_text = [];
    $flowchart = new Flowchart(default_edge_style: $options['default_edge_style']);
    foreach ($graph->getEdges() as $edge) {
      array_push($mermaid_graph_text, ...$flowchart->getEdgeText($edge));
    }
    return $mermaid_graph_text;
  }

}
