<?php

namespace Drupal\mermaid_graphapi\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\graphapi\Plugin\views\style\Graph;
use Drupal\mermaid_graphapi\Mermaid\Flowchart;

/**
 * Style plugin to render a view as a Mermaid graph.
 *
 * @ViewsStyle(
 *   id = "mermaid_graph",
 *   title = @Translation("Mermaid Graph"),
 *   help = @Translation("Displays data in a Mermaid graph."),
 *   theme = "mermaid_view_graph",
 *   display_types = {"normal"}
 * )
 */
class MermaidGraph extends Graph {

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    // We expect some results with empty IDs.
    $options['warn_empty_id'] = ['default' => FALSE];

    // Use the global default.
    $options['graph_direction'] = ['default' => 'default'];

    $options['link_to_entity'] = ['default' => TRUE];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineMapping() {
    $mapping = parent::defineMapping();
    $supported = implode(', ', array_keys(Flowchart::getVertexShapesInfo()));
    $mapping['vertex_shape'] = [
      '#title' => $this->t('Vertex shape'),
      '#description' => $this->t("The shape to use for each vertex (node).<br />Valid values are: {$supported}.<br />This is not supported by all formats."),
    ];
    $supported = implode(', ', array_keys(Flowchart::getEdgeStylesInfo()));
    $mapping['edge_style'] = [
      '#title' => $this->t('Edge style'),
      '#description' => $this->t("The style to use for each edge (link).<br />Valid values are: {$supported}.<br />This is not supported by all formats."),
    ];

    return $mapping;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $graph_formats = $this->graphFormatManager->getDefinitions();
    // This style plugin should only handle Mermaid graphs.
    foreach ($graph_formats as $format_id => $format) {
      if (!array_key_exists($format_id, $form['format_id']['#options'])) {
        continue;
      }
      if ($format['engine'] === 'mermaid') {
        continue;
      }
      unset($form['format_id']['#options'][$format_id]);
    }
    // Clean up unused form elements.
    unset($form['directed']);
    unset($form['width']);
    unset($form['height']);
    unset($form['warn_empty_id']);
    unset($form['warn_multiple_id']);

    $direction_options = ['default' => 'Use site-wide default'];

    foreach (Flowchart::getGraphDirectionsInfo() as $direction => $info) {
      $direction_options[$direction] = $this->t($info['label']);
    }
    $form['graph_direction'] = [
      '#type' => 'radios',
      '#title' => $this->t('Graph direction'),
      '#options' => $direction_options,
      '#default_value' => $this->options['graph_direction'],
    ];

    $form['link_to_entity'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Link vertices to entities'),
      '#default_value' => $this->options['link_to_entity'],
    ];
  }

}
