<?php

namespace Drupal\message_filter\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteProviderInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure Message Filter settings for this site.
 */
class MessageFilterSettingsForm extends ConfigFormBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The route provider.
   *
   * @var \Drupal\Core\Routing\RouteProviderInterface
   */
  protected $routeProvider;

  /**
   * Constructs a MessageFilterSettingsForm object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The factory for configuration objects.
   * @param \Drupal\Core\Config\TypedConfigManagerInterface $typed_config_manager
   *   The typed config manager.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Routing\RouteProviderInterface $route_provider
   *   The route provider.
   */
  public function __construct(ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typed_config_manager, EntityTypeManagerInterface $entity_type_manager, RouteProviderInterface $route_provider) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
    $this->routeProvider = $route_provider;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('router.route_provider')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'message_filter_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['message_filter.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('message_filter.settings');

    // Add debug information.
    $form['debug'] = [
      '#type' => 'details',
      '#title' => $this->t('Debug Information'),
      '#open' => FALSE,
    ];

    $config_data = $config->getRawData();
    $form['debug']['current_config'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Current Configuration'),
      '#default_value' => json_encode($config_data, JSON_PRETTY_PRINT),
      '#rows' => 10,
      '#attributes' => ['readonly' => 'readonly'],
    ];

    $form['debug']['config_summary'] = [
      '#type' => 'item',
      '#title' => $this->t('Configuration Summary'),
      '#markup' => $this->t('Enabled: @enabled | Role rules count: @count', [
        '@enabled' => $config->get('enabled') ? 'Yes' : 'No',
        '@count' => count($config->get('role_rules') ?: []),
      ]),
    ];

    $form['general'] = [
      '#type' => 'details',
      '#title' => $this->t('General Settings'),
      '#open' => TRUE,
    ];

    $form['general']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable message filtering'),
      '#description' => $this->t('When enabled, messages will be filtered according to the rules below.'),
      '#default_value' => $config->get('enabled'),
    ];

    // Get all roles for the configuration sections.
    $roles = $this->entityTypeManager->getStorage('user_role')->loadMultiple();
    $role_rules = $config->get('role_rules') ?: [];

    $message_type_options = [
      MessengerInterface::TYPE_STATUS => $this->t('Status'),
      MessengerInterface::TYPE_WARNING => $this->t('Warning'),
      MessengerInterface::TYPE_ERROR => $this->t('Error'),
    ];

    $form['role_rules'] = [
      '#type' => 'details',
      '#title' => $this->t('Role-based Filtering Rules'),
      '#description' => $this->t('Configure specific filtering rules for each role.'),
      '#open' => TRUE,
      '#tree' => TRUE,
    ];

    foreach ($roles as $role_id => $role) {
      // Skip anonymous and authenticated roles if needed
      if (in_array($role_id, ['anonymous', 'authenticated', 'administrator'])) {
        continue;
      }

      $form['role_rules'][$role_id] = [
        '#type' => 'details',
        '#title' => $this->t('Rules for @role', ['@role' => $role->label()]),
        '#open' => FALSE,
      ];

      $form['role_rules'][$role_id]['enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable filtering for @role', ['@role' => $role->label()]),
        '#default_value' => $role_rules[$role_id]['enabled'] ?? FALSE,
      ];

      $form['role_rules'][$role_id]['block_all_messages'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Block ALL messages'),
        '#description' => $this->t('When checked, ALL messages will be blocked for users with the @role role, regardless of type or route.', ['@role' => $role->label()]),
        '#default_value' => $role_rules[$role_id]['block_all_messages'] ?? FALSE,
        '#states' => [
          'visible' => [
            ':input[name="role_rules[' . $role_id . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];

      $form['role_rules'][$role_id]['blocked_message_types'] = [
        '#type' => 'checkboxes',
        '#title' => $this->t('Blocked message types'),
        '#description' => $this->t('Select which message types should be filtered for users with the @role role.', ['@role' => $role->label()]),
        '#options' => $message_type_options,
        '#default_value' => $role_rules[$role_id]['blocked_message_types'] ?? [],
        '#states' => [
          'visible' => [
            ':input[name="role_rules[' . $role_id . '][enabled]"]' => ['checked' => TRUE],
            ':input[name="role_rules[' . $role_id . '][block_all_messages]"]' => ['checked' => FALSE],
          ],
        ],
      ];

      $form['role_rules'][$role_id]['blocked_routes'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Blocked routes'),
        '#description' => $this->t('Enter route names (one per line) where messages should be filtered for @role users. Examples: user.login, user.register, node.add, entity.node.canonical', ['@role' => $role->label()]),
        '#default_value' => implode("\n", $role_rules[$role_id]['blocked_routes'] ?? []),
        '#rows' => 6,
        '#states' => [
          'visible' => [
            ':input[name="role_rules[' . $role_id . '][enabled]"]' => ['checked' => TRUE],
            ':input[name="role_rules[' . $role_id . '][block_all_messages]"]' => ['checked' => FALSE],
          ],
        ],
      ];

      $form['role_rules'][$role_id]['blocked_urls'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Blocked URLs'),
        '#description' => $this->t('Enter URL paths (one per line) where messages should be filtered for @role users. Examples: /admin/config, /user/login, /node/add/article, /node/*/edit', ['@role' => $role->label()]),
        '#default_value' => implode("\n", $role_rules[$role_id]['blocked_urls'] ?? []),
        '#rows' => 6,
        '#states' => [
          'visible' => [
            ':input[name="role_rules[' . $role_id . '][enabled]"]' => ['checked' => TRUE],
            ':input[name="role_rules[' . $role_id . '][block_all_messages]"]' => ['checked' => FALSE],
          ],
        ],
      ];

      $form['role_rules'][$role_id]['priority'] = [
        '#type' => 'number',
        '#title' => $this->t('Priority'),
        '#description' => $this->t('Priority for this role rule (higher number = higher priority). Used when a user has multiple roles.'),
        '#default_value' => $role_rules[$role_id]['priority'] ?? 0,
        '#min' => 0,
        '#max' => 100,
        '#states' => [
          'visible' => [
            ':input[name="role_rules[' . $role_id . '][enabled]"]' => ['checked' => TRUE],
          ],
        ],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $role_rules_values = $form_state->getValue('role_rules', []);

    foreach ($role_rules_values as $role_id => $role_data) {
      if (!empty($role_data['enabled'])) {
        // Validate routes format.
        if (!empty($role_data['blocked_routes'])) {
          $routes = explode("\n", $role_data['blocked_routes']);
          foreach ($routes as $line_number => $route) {
            $route = trim($route);
            if (!empty($route)) {
              // Basic validation: route names should not contain spaces or special characters except dots and underscores.
              if (!preg_match('/^[a-zA-Z0-9_.]+$/', $route)) {
                $form_state->setError(
                  $form['role_rules'][$role_id]['blocked_routes'],
                  $this->t('Invalid route name "@route" on line @line. Route names should only contain letters, numbers, dots, and underscores.', [
                    '@route' => $route,
                    '@line' => $line_number + 1,
                  ])
                );
              }
            }
          }
        }

        // Validate URLs format.
        if (!empty($role_data['blocked_urls'])) {
          $urls = explode("\n", $role_data['blocked_urls']);
          foreach ($urls as $line_number => $url) {
            $url = trim($url);
            if (!empty($url)) {
              // Basic validation: URLs should start with / and contain valid path characters
              if (!preg_match('/^\/[a-zA-Z0-9\/_\-\*\.]*$/', $url)) {
                $form_state->setError(
                  $form['role_rules'][$role_id]['blocked_urls'],
                  $this->t('Invalid URL path "@url" on line @line. URLs should start with "/" and contain only valid path characters (letters, numbers, /, -, _, *, .).', [
                    '@url' => $url,
                    '@line' => $line_number + 1,
                  ])
                );
              }
            }
          }
        }

        // Validate priority range.
        $priority = $role_data['priority'] ?? 0;
        if ($priority < 0 || $priority > 100) {
          $form_state->setError(
            $form['role_rules'][$role_id]['priority'],
            $this->t('Priority must be between 0 and 100.')
          );
        }
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // Check permissions.
    if (!\Drupal::currentUser()->hasPermission('administer message filter')) {
      \Drupal::logger('message_filter')->error('User @uid does not have permission to administer message filter', [
        '@uid' => \Drupal::currentUser()->id(),
      ]);
      $this->messenger()->addError($this->t('You do not have permission to save these settings.'));
      return;
    }

    $config = $this->config('message_filter.settings');

    // Save general settings.
    $enabled = $form_state->getValue('enabled');

    \Drupal::logger('message_filter')->debug('Saving general settings - Enabled: @enabled', [
      '@enabled' => $enabled ? 'TRUE' : 'FALSE',
    ]);

    $config
      ->set('enabled', $enabled);

    // Process role rules.
    $role_rules_values = $form_state->getValue('role_rules', []);
    $role_rules = [];

    \Drupal::logger('message_filter')->debug('Processing @count role rules', [
      '@count' => is_array($role_rules_values) ? count($role_rules_values) : 0,
    ]);

    if (is_array($role_rules_values)) {
      foreach ($role_rules_values as $role_id => $role_data) {
        if (is_array($role_data) && !empty($role_data['enabled'])) {
          // Process blocked routes for this role.
          $blocked_routes = [];
          if (!empty($role_data['blocked_routes'])) {
            $blocked_routes = array_filter(
              array_map('trim', explode("\n", $role_data['blocked_routes']))
            );
          }

          // Process blocked URLs for this role.
          $blocked_urls = [];
          if (!empty($role_data['blocked_urls'])) {
            $blocked_urls = array_filter(
              array_map('trim', explode("\n", $role_data['blocked_urls']))
            );
          }

          // Process blocked message types for this role.
          $blocked_message_types = [];
          if (!empty($role_data['blocked_message_types']) && is_array($role_data['blocked_message_types'])) {
            $blocked_message_types = array_filter($role_data['blocked_message_types']);
          }

          $role_rules[$role_id] = [
            'enabled' => TRUE,
            'block_all_messages' => !empty($role_data['block_all_messages']),
            'blocked_routes' => array_values($blocked_routes),
            'blocked_urls' => array_values($blocked_urls),
            'blocked_message_types' => array_values($blocked_message_types),
            'priority' => (int) ($role_data['priority'] ?? 0),
          ];

          \Drupal::logger('message_filter')->debug('Added rule for role @role_id', ['@role_id' => $role_id]);
        }
      }
    }

    \Drupal::logger('message_filter')->debug('Processed @count active role rules', [
      '@count' => count($role_rules),
    ]);

    try {
      $config->set('role_rules', $role_rules);
      $result = $config->save();

      if ($result) {
        \Drupal::logger('message_filter')->info('Configuration saved successfully');
        $this->messenger()->addStatus($this->t('The configuration has been saved.'));
      } else {
        \Drupal::logger('message_filter')->error('Failed to save configuration');
        $this->messenger()->addError($this->t('Failed to save the configuration.'));
      }
    } catch (\Exception $e) {
      \Drupal::logger('message_filter')->error('Exception saving configuration: @message', [
        '@message' => $e->getMessage(),
      ]);
      $this->messenger()->addError($this->t('Error saving configuration: @message', ['@message' => $e->getMessage()]));
    }

    parent::submitForm($form, $form_state);
  }

}
