<?php

namespace Drupal\message_filter\Messenger;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\message_filter\Plugin\MessageBypassStrategyManager;
use Drupal\message_filter\Service\UnfilteredMessengerService;
use Psr\Log\LoggerInterface;

/**
 * Decorator for MessengerInterface that filters messages based on configuration.
 */
class FilteredMessenger implements MessengerInterface {

  /**
   * The decorated messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected $messenger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The unfiltered messenger service.
   *
   * @var \Drupal\message_filter\Service\UnfilteredMessengerService
   */
  protected $unfilteredMessenger;

  /**
   * The bypass strategy manager.
   *
   * @var \Drupal\message_filter\Plugin\MessageBypassStrategyManager
   */
  protected $bypassStrategyManager;

  /**
   * Constructs a FilteredMessenger object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The decorated messenger service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\message_filter\Service\UnfilteredMessengerService $unfiltered_messenger
   *   The unfiltered messenger service.
   * @param \Drupal\message_filter\Plugin\MessageBypassStrategyManager $bypass_strategy_manager
   *   The bypass strategy manager.
   */
  public function __construct(
    MessengerInterface $messenger,
    AccountInterface $current_user,
    ConfigFactoryInterface $config_factory,
    RouteMatchInterface $route_match,
    LoggerInterface $logger,
    UnfilteredMessengerService $unfiltered_messenger,
    MessageBypassStrategyManager $bypass_strategy_manager
  ) {
    $this->messenger = $messenger;
    $this->currentUser = $current_user;
    $this->configFactory = $config_factory;
    $this->routeMatch = $route_match;
    $this->logger = $logger;
    $this->unfilteredMessenger = $unfiltered_messenger;
    $this->bypassStrategyManager = $bypass_strategy_manager;
  }

  /**
   * {@inheritdoc}
   */
  public function addMessage($message, $type = self::TYPE_STATUS, $repeat = FALSE) {
    if (!$this->shouldBlockMessage($message, $type)) {
      return $this->messenger->addMessage($message, $type, $repeat);
    }

    $this->logBlockedMessage($message, $type);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function addStatus($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_STATUS, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addError($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_ERROR, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addWarning($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_WARNING, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function all() {
    return $this->messenger->all();
  }

  /**
   * {@inheritdoc}
   */
  public function messagesByType($type) {
    return $this->messenger->messagesByType($type);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll() {
    return $this->messenger->deleteAll();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteByType($type) {
    return $this->messenger->deleteByType($type);
  }

  /**
   * Determines if a message should be blocked.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if the message should be blocked, FALSE otherwise.
   */
  protected function shouldBlockMessage($message, $type) {
    $config = $this->configFactory->get('message_filter.settings');

    // If filtering is disabled, don't block anything.
    if (!$config->get('enabled')) {
      return FALSE;
    }

    // PRIORITY 1: Check if message is from unfiltered messenger - NEVER block these
    if ($this->unfilteredMessenger->isUnfilteredMessage($message, $type)) {
      return FALSE;
    }

    // PRIORITY 2: Check bypass permission BEFORE applying role rules
    if ($this->currentUser->hasPermission('bypass message filter')) {
      return FALSE;
    }

    // Get role-based rules AFTER checking bypass permission
    $role_rules = $config->get('role_rules') ?: [];
    $user_roles = $this->currentUser->getRoles();

    // Check if any of the user's roles have explicit filtering rules
    $has_explicit_rules = FALSE;
    $applicable_rules = [];
    foreach ($user_roles as $role_id) {
      if (isset($role_rules[$role_id]) && $role_rules[$role_id]['enabled']) {
        $has_explicit_rules = TRUE;
        $applicable_rules[] = $role_rules[$role_id];
      }
    }

    // If we have explicit role rules, apply them
    if ($has_explicit_rules) {
      // Sort by priority (highest first).
      usort($applicable_rules, function($a, $b) {
        return ($b['priority'] ?? 0) - ($a['priority'] ?? 0);
      });

      $current_route = $this->routeMatch->getRouteName();

      // Apply the first (highest priority) applicable rule.
      foreach ($applicable_rules as $rule) {
        // If "block all messages" is enabled for this rule, block everything.
        if (!empty($rule['block_all_messages'])) {
          return TRUE;
        }

        $blocked_routes = $rule['blocked_routes'] ?? [];
        $blocked_urls = $rule['blocked_urls'] ?? [];
        $blocked_types = $rule['blocked_message_types'] ?? [];

        // Get current URL path for URL matching
        $current_path = \Drupal::request()->getPathInfo();

        // Check if current route or URL should trigger filtering
        $route_or_url_matches = FALSE;

        // Check blocked routes
        if (!empty($blocked_routes) && in_array($current_route, $blocked_routes)) {
          $route_or_url_matches = TRUE;
        }

        // Check blocked URLs (with wildcard support)
        if (!empty($blocked_urls)) {
          foreach ($blocked_urls as $blocked_url) {
            if ($this->urlMatches($current_path, $blocked_url)) {
              $route_or_url_matches = TRUE;
              break;
            }
          }
        }

        // If blocked_routes or blocked_urls is specified, only apply filtering when they match
        if (!empty($blocked_routes) || !empty($blocked_urls)) {
          if ($route_or_url_matches) {
            // We're on a blocked route/URL, now check if message type should be blocked
            if (!empty($blocked_types) && in_array($type, $blocked_types)) {
              return TRUE;
            }
          }
        } else {
          // No specific routes defined, apply message type filtering globally
          if (!empty($blocked_types) && in_array($type, $blocked_types)) {
            return TRUE;
          }
        }
      }
    }

    // Check bypass strategies.
    foreach ($this->bypassStrategyManager->getDefinitions() as $plugin_id => $definition) {
      $strategy = $this->bypassStrategyManager->createInstance($plugin_id);
      if ($strategy->shouldBypass($message, $type, $this->currentUser, $this->routeMatch)) {
        return FALSE;
      }
    }

    return FALSE;
  }


  /**
   * Logs a blocked message for debugging purposes.
   *
   * @param mixed $message
   *   The blocked message.
   * @param string $type
   *   The message type.
   */
  protected function logBlockedMessage($message, $type) {
    // Always log blocked messages in debug mode for troubleshooting
    $this->logger->debug('Blocked @type message: @message', [
      '@type' => $type,
      '@message' => is_string($message) ? $message : serialize($message),
    ]);
  }

  /**
   * Check if a URL path matches a pattern (with wildcard support).
   *
   * @param string $current_path
   *   The current URL path.
   * @param string $pattern
   *   The pattern to match against (supports * wildcards).
   *
   * @return bool
   *   TRUE if the path matches the pattern, FALSE otherwise.
   */
  protected function urlMatches($current_path, $pattern) {
    // Exact match
    if ($current_path === $pattern) {
      return TRUE;
    }

    // Convert wildcard pattern to regex
    if (strpos($pattern, '*') !== FALSE) {
      // Escape special regex characters except *
      $escaped_pattern = preg_quote($pattern, '/');
      // Replace escaped \* with .*
      $regex_pattern = str_replace('\*', '.*', $escaped_pattern);
      // Add anchors for full match
      $regex_pattern = '/^' . $regex_pattern . '$/';

      return preg_match($regex_pattern, $current_path) === 1;
    }

    return FALSE;
  }

}
