<?php

declare(strict_types=1);

namespace Drupal\Tests\message_filter\Unit\Service;

use Drupal\message_filter\Service\UnfilteredMessengerService;
use Drupal\Tests\UnitTestCase;
use Psr\Log\LoggerInterface;

/**
 * Comprehensive tests for UnfilteredMessengerService.
 *
 * @group message_filter
 * @coversDefaultClass \Drupal\message_filter\Service\UnfilteredMessengerService
 */
final class UnfilteredMessengerServiceTest extends UnitTestCase {

  private UnfilteredMessengerService $service;
  private LoggerInterface $logger;

  protected function setUp(): void {
    parent::setUp();
    $this->logger = $this->createMock(LoggerInterface::class);
    $this->service = new UnfilteredMessengerService($this->logger);
  }

  /**
   * Data provider for message types.
   */
  public static function messageTypeProvider(): array {
    return [
      ['status'],
      ['warning'],
      ['error'],
      ['custom_type'],
    ];
  }

  /**
   * Data provider for message content.
   */
  public static function messageContentProvider(): array {
    return [
      'string message' => ['Hello World'],
      'empty string' => [''],
      'null value' => [null],
      'integer' => [42],
      'boolean true' => [true],
      'boolean false' => [false],
      'simple array' => [['key' => 'value']],
      'complex array' => [['nested' => ['deep' => ['value' => 123]]]],
    ];
  }

  /**
   * Tests core message functionality without display.
   *
   * @covers ::addMessage
   * @covers ::getUnfilteredMessages
   * @covers ::generateMessageHash
   * @dataProvider messageTypeProvider
   */
  public function testAddMessageCore(string $type): void {
    $this->logger->method('debug');

    $result = $this->service->addMessage('Test message', $type, [], false);

    $this->assertSame($this->service, $result);

    $messages = $this->service->getUnfilteredMessages();
    $this->assertCount(1, $messages);

    $message_data = reset($messages);
    $this->assertSame('Test message', $message_data['message']);
    $this->assertSame($type, $message_data['type']);
    $this->assertIsInt($message_data['timestamp']);
  }

  /**
   * Tests message content handling.
   *
   * @covers ::addMessage
   * @covers ::isUnfilteredMessage
   * @dataProvider messageContentProvider
   */
  public function testMessageContent($content): void {
    $this->logger->method('debug');

    $this->service->addMessage($content, 'status', [], false);
    $this->assertTrue($this->service->isUnfilteredMessage($content, 'status'));
  }

  /**
   * Tests message uniqueness and hash collision handling.
   *
   * @covers ::generateMessageHash
   */
  public function testMessageUniqueness(): void {
    $this->logger->method('debug');

    // Same message, same type - should overwrite
    $this->service->addMessage('Duplicate', 'status', [], false);
    $this->service->addMessage('Duplicate', 'status', [], false);
    $this->assertCount(1, $this->service->getUnfilteredMessages());

    // Same message, different type - should be separate
    $this->service->addMessage('Duplicate', 'warning', [], false);
    $this->assertCount(2, $this->service->getUnfilteredMessages());

    // Different message, same type - should be separate
    $this->service->addMessage('Different', 'status', [], false);
    $this->assertCount(3, $this->service->getUnfilteredMessages());
  }

  /**
   * Tests all utility methods.
   *
   * @covers ::isUnfilteredMessage
   * @covers ::isCustomMessage
   * @covers ::removeMessage
   * @covers ::clearUnfilteredMessages
   * @covers ::clearCustomMessages
   * @covers ::getCustomMessages
   */
  public function testAllUtilityMethods(): void {
    $this->logger->method('debug');

    // Add test data
    $this->service->addMessage('Message 1', 'status', [], false);
    $this->service->addMessage('Message 2', 'warning', [], false);

    // Test checking methods
    $this->assertTrue($this->service->isUnfilteredMessage('Message 1', 'status'));
    $this->assertTrue($this->service->isCustomMessage('Message 1', 'status'));
    $this->assertFalse($this->service->isUnfilteredMessage('Nonexistent', 'status'));

    // Test aliases return same results
    $this->assertSame(
      $this->service->getUnfilteredMessages(),
      $this->service->getCustomMessages()
    );

    // Test removal
    $this->service->removeMessage('Message 1', 'status');
    $this->assertFalse($this->service->isUnfilteredMessage('Message 1', 'status'));
    $this->assertCount(1, $this->service->getUnfilteredMessages());

    // Test clearing
    $this->service->clearUnfilteredMessages();
    $this->assertCount(0, $this->service->getUnfilteredMessages());

    // Test alias clearing
    $this->service->addMessage('New message', 'error', [], false);
    $this->assertCount(1, $this->service->getUnfilteredMessages());
    $this->service->clearCustomMessages();
    $this->assertCount(0, $this->service->getUnfilteredMessages());
  }

  /**
   * Tests timestamp and context handling.
   *
   * @covers ::addMessage
   */
  public function testTimestampAndContext(): void {
    $this->logger->method('debug');

    $context = ['user_id' => 123, 'session' => 'abc123'];
    $before_time = time();

    $this->service->addMessage('Timestamped message', 'status', $context, false);

    $after_time = time();
    $messages = $this->service->getUnfilteredMessages();
    $message_data = reset($messages);

    // Verify timestamp is within expected range
    $this->assertGreaterThanOrEqual($before_time, $message_data['timestamp']);
    $this->assertLessThanOrEqual($after_time, $message_data['timestamp']);

    // Verify context is preserved
    $this->assertSame($context, $message_data['context']);
  }

  /**
   * Tests method chaining.
   *
   * @covers ::addMessage
   * @covers ::removeMessage
   * @covers ::clearUnfilteredMessages
   */
  public function testMethodChaining(): void {
    $this->logger->method('debug');

    $result = $this->service
      ->addMessage('Message 1', 'status', [], false)
      ->addMessage('Message 2', 'warning', [], false)
      ->removeMessage('Message 1', 'status')
      ->clearUnfilteredMessages();

    $this->assertSame($this->service, $result);
    $this->assertCount(0, $this->service->getUnfilteredMessages());
  }

}
