# Message Filter

Un module Drupal 10/11 avancé pour filtrer les messages système (status, warning, error) basé sur des règles configurables par rôle.

## Qu'est-ce que Message Filter ?

**Message Filter** résout un problème courant dans les sites Drupal : **la surcharge de messages système** qui peuvent polluer l'expérience utilisateur, particulièrement pour les utilisateurs non-administrateurs.

### Le problème résolu

Dans un site Drupal standard, tous les messages système (confirmations, avertissements, erreurs) sont affichés à tous les utilisateurs sans distinction. Cela peut créer :

- **Confusion utilisateur** : Messages techniques incompréhensibles pour les utilisateurs finaux
- **Pollution visuelle** : Trop de notifications masquent les informations importantes
- **Sécurité** : Exposition d'informations sensibles (chemins, erreurs SQL, etc.)
- **UX dégradée** : Interface encombrée pour les rôles non-techniques

### La solution apportée

Message Filter introduit un **système de filtrage intelligent** qui permet de :

1. **Contrôler l'affichage par rôle** : Les administrateurs voient tout, les éditeurs voient l'essentiel, les utilisateurs finaux ne voient que ce qui les concerne
2. **Filtrer par contexte** : Masquer certains messages sur des pages spécifiques (/admin/*, /node/add, etc.)
3. **Prioriser les messages** : Système de bypass pour les messages critiques qui doivent TOUJOURS être affichés
4. **Personnaliser l'expérience** : Plugins extensibles pour des règles métier complexes

### Cas d'usage typiques

- **Site d'entreprise** : Masquer les messages techniques aux employés, garder les notifications métier
- **Site e-commerce** : Filtrer les erreurs système pour les clients, les montrer aux gestionnaires
- **Plateforme éducative** : Adapter l'affichage selon le niveau d'expertise (étudiant vs professeur)
- **Site communautaire** : Réduire le bruit pour les contributeurs occasionnels

### Exemple concret

**Avant Message Filter :**

```text
❌ "Configuration object updated successfully" (visible à tous)
❌ "PHP Notice: Undefined index in custom_module.module line 42" (visible aux utilisateurs)
❌ "Cache cleared" (affiché même aux visiteurs anonymes)
```

**Après Message Filter :**

```text
✓ Administrateur : Voit tous les messages (debug, erreurs, confirmations)
✓ Éditeur : Voit uniquement les messages liés à son travail (contenu publié, erreurs de formulaire)
✓ Utilisateur : Voit uniquement les confirmations importantes (commande validée, profil mis à jour)
✓ Anonyme : Aucun message technique, seulement les notifications essentielles
```

## Vue d'ensemble

Le module **Message Filter** utilise le pattern Decorator pour intercepter et filtrer les messages Drupal avant leur affichage, offrant un contrôle granulaire basé sur :

- **Rôles utilisateur** avec règles personnalisées
- **Routes spécifiques** et patterns d'URL
- **Types de messages** (status, warning, error)
- **Système de plugins** pour des stratégies de contournement

## Fonctionnalités

### Filtrage Avancé

- **Filtrage par rôle** : Règles spécifiques par rôle utilisateur
- **Filtrage par route** : Ciblage de pages/routes spécifiques
- **Filtrage par URL** : Support des wildcards (/admin/*)
- **Filtrage par type** : Status, warning, error sélectivement
- **Système de priorités** : Gestion des conflits entre règles

### Système de Contournement

- **Permission de bypass** : bypass message filter
- **Messages non filtrés** : API pour messages prioritaires
- **Plugins de stratégie** : Contournement conditionnel extensible
- **Système d'annotations** : Configuration déclarative des plugins

### Interface d'Administration

- **Interface moderne** : Configuration intuitive par rôle
- **Prévisualisation** : Debug et validation des règles
- **Export/Import** : Configuration via YAML

## Architecture

### Services Principaux

Le module s'appuie sur 3 services clés :

1. **messenger.filtered** : Décorateur principal du service messenger
2. **message_filter.unfiltered_messenger** : Service pour messages prioritaires
3. **message_filter.bypass_strategy_manager** : Manager des plugins de contournement

### Classes Principales

| Classe | Responsabilité |
|--------|----------------|
| FilteredMessenger | Décorateur principal, logique de filtrage |
| UnfilteredMessengerService | API pour messages prioritaires |
| MessageBypassStrategyManager | Gestion des plugins de contournement |
| MessageFilterSettingsForm | Interface d'administration |
| DemoBypassStrategy | Plugin exemple de contournement |

## Installation

### Prérequis

- **Drupal** : 10.x ou 11.x
- **PHP** : 8.1+

### Installation manuelle

1. Placez le module dans modules/custom/message_filter
2. Activez via l'interface : /admin/modules
3. Configurez via : /admin/config/system/message-filter

## Configuration

### Configuration de base

1. **Accéder à la configuration** : /admin/config/system/message-filter
2. **Activer le filtrage** : Cochez "Enable message filtering"
3. **Configurer les rôles** : Définissez des règles par rôle

### Permissions

| Permission | Description |
|------------|-------------|
| administer message filter | Accès à la configuration |
| bypass message filter | Contourner tous les filtres |

## API de Développement

### Message Flow Architecture

```text
┌─────────────────┐    ┌──────────────────┐    ┌─────────────────┐
│   Application   │───▶│ FilteredMessenger │───▶│  Core Messenger │
│   addMessage()  │    │    (Decorator)    │    │   (Decorated)   │
└─────────────────┘    └──────────────────┘    └─────────────────┘
                                │
                                ▼
                       ┌──────────────────┐
                       │ shouldBlockMessage│
                       │    (Logic)       │
                       └──────────────────┘
                                │
                    ┌───────────┼───────────┐
                    ▼           ▼           ▼
           ┌─────────────┐ ┌─────────┐ ┌──────────┐
           │ Unfiltered  │ │ Bypass  │ │   Role   │
           │   Check     │ │Permission│ │  Rules   │
           └─────────────┘ └─────────┘ └──────────┘
```

### Service Decoration Pattern

Le module utilise le pattern Decorator pour intercepter les appels au service `messenger` :

```yaml
# message_filter.services.yml
messenger.filtered:
  class: Drupal\message_filter\Messenger\FilteredMessenger
  decorates: messenger
  decoration_priority: 5
  arguments:
    - '@messenger.filtered.inner'  # Service original
    - '@current_user'
    - '@config.factory'
    - '@current_route_match'
    - '@logger.channel.message_filter'
    - '@message_filter.unfiltered_messenger'
    - '@message_filter.bypass_strategy_manager'
```

### Algorithme de Filtrage

```php
/**
 * Ordre de priorité dans shouldBlockMessage() :
 *
 * 1. PRIORITY 1: Messages UnfilteredMessenger (JAMAIS bloqués)
 * 2. PRIORITY 2: Permission 'bypass message filter'
 * 3. PRIORITY 3: Règles par rôle (selon priorité configurée)
 * 4. PRIORITY 4: Plugins bypass strategies (selon weight)
 * 5. DEFAULT: Pas de blocage
 */
protected function shouldBlockMessage($message, $type) {
  // 1. Check unfiltered messages (absolute priority)
  if ($this->unfilteredMessenger->isUnfilteredMessage($message, $type)) {
    return FALSE;
  }

  // 2. Check bypass permission
  if ($this->currentUser->hasPermission('bypass message filter')) {
    return FALSE;
  }

  // 3. Apply role-based rules with priority sorting
  $applicable_rules = $this->getApplicableRulesByPriority();
  foreach ($applicable_rules as $rule) {
    if ($this->ruleApplies($rule, $message, $type)) {
      return TRUE;
    }
  }

  // 4. Check bypass strategies
  foreach ($this->bypassStrategyManager->getDefinitions() as $plugin_id => $definition) {
    $strategy = $this->bypassStrategyManager->createInstance($plugin_id);
    if ($strategy->shouldBypass($message, $type, $this->currentUser, $this->routeMatch)) {
      return FALSE;
    }
  }

  return FALSE;
}
```

### Messages non filtrés

Pour des messages critiques qui ne doivent jamais être filtrés :

```php
// Via le service
$unfiltered_messenger = \Drupal::service('message_filter.unfiltered_messenger');
$unfiltered_messenger->addMessage('Message critique', 'error');

// Via l'injection de dépendance
public function __construct(UnfilteredMessengerService $unfiltered_messenger) {
  $this->unfilteredMessenger = $unfiltered_messenger;
}

public function myMethod() {
  // Message qui ne sera JAMAIS filtré
  $this->unfilteredMessenger->addMessage('Message prioritaire', 'status');
}
```

### Création de Plugin Bypass Strategy

Créez des stratégies de contournement avancées :

```php
<?php

namespace Drupal\my_module\Plugin\MessageBypassStrategy;

use Drupal\Core\Plugin\PluginBase;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\message_filter\Plugin\MessageBypassStrategyInterface;

/**
 * Advanced bypass strategy for specific business logic.
 *
 * @MessageBypassStrategy(
 *   id = "advanced_bypass",
 *   label = @Translation("Advanced Bypass Strategy"),
 *   description = @Translation("Complex bypass logic for enterprise needs."),
 *   weight = 15
 * )
 */
class AdvancedBypassStrategy extends PluginBase implements MessageBypassStrategyInterface {

  /**
   * {@inheritdoc}
   */
  public function shouldBypass($message, $type, AccountInterface $account, RouteMatchInterface $route_match) {
    // 1. VIP Users bypass
    if ($account->hasRole('vip_user')) {
      return TRUE;
    }

    // 2. Emergency messages bypass
    if (is_string($message) && preg_match('/URGENT|CRITICAL|EMERGENCY/i', $message)) {
      \Drupal::logger('message_filter')->notice('Emergency bypass for: @message', ['@message' => $message]);
      return TRUE;
    }

    // 3. Maintenance mode bypass
    if (\Drupal::state()->get('system.maintenance_mode')) {
      return TRUE;
    }

    // 4. Specific routes bypass
    $bypass_routes = [
      'system.site_maintenance_mode',
      'user.login',
      'user.logout',
    ];
    if (in_array($route_match->getRouteName(), $bypass_routes)) {
      return TRUE;
    }

    // 5. Time-based bypass (working hours)
    $current_hour = (int) date('H');
    if ($current_hour < 9 || $current_hour > 17) {
      // Outside working hours, show all messages
      return TRUE;
    }

    // 6. IP-based bypass for admins
    $request = \Drupal::request();
    $admin_ips = ['192.168.1.100', '10.0.0.50'];
    if (in_array($request->getClientIp(), $admin_ips)) {
      return TRUE;
    }

    return FALSE;
  }
}
```

### Hook Integration

Intégrez avec les hooks Drupal pour un contrôle avancé :

```php
// my_module.module

/**
 * Implements hook_message_filter_bypass_alter().
 */
function my_module_message_filter_bypass_alter(&$should_bypass, $message, $type, $account, $route_match) {
  // Custom logic to override bypass decision
  if ($account->id() == 1) {
    $should_bypass = TRUE; // User 1 always bypasses
  }

  // Log bypass decisions for audit
  if ($should_bypass) {
    \Drupal::logger('my_module')->info('Message bypass granted for user @uid: @message', [
      '@uid' => $account->id(),
      '@message' => is_string($message) ? $message : serialize($message),
    ]);
  }
}

/**
 * Implements hook_message_filter_rules_alter().
 */
function my_module_message_filter_rules_alter(&$rules, $account) {
  // Dynamically modify rules based on business logic
  if ($account->hasRole('premium_user')) {
    // Premium users see fewer filtered messages
    foreach ($rules as &$rule) {
      $rule['priority'] = ($rule['priority'] ?? 0) - 5;
    }
  }
}
```

### Configuration Programmatique

Gérez la configuration via code :

```php
// Configuration service
$config = \Drupal::configFactory()->getEditable('message_filter.settings');

// Enable filtering
$config->set('enabled', TRUE);

// Add role rule programmatically
$role_rules = $config->get('role_rules') ?: [];
$role_rules['editor'] = [
  'enabled' => TRUE,
  'priority' => 10,
  'blocked_message_types' => ['status', 'warning'],
  'blocked_routes' => ['node.add', 'node.edit_form'],
  'blocked_urls' => ['/admin/content/*'],
  'block_all_messages' => FALSE,
];

$config->set('role_rules', $role_rules);
$config->save();

// Clear caches to apply changes
drupal_flush_all_caches();
```

### Testing Strategies

Testez vos plugins et configurations :

```php
<?php

namespace Drupal\Tests\my_module\Unit\Plugin\MessageBypassStrategy;

use Drupal\my_module\Plugin\MessageBypassStrategy\AdvancedBypassStrategy;
use Drupal\Tests\UnitTestCase;

class AdvancedBypassStrategyTest extends UnitTestCase {

  private AdvancedBypassStrategy $strategy;

  protected function setUp(): void {
    parent::setUp();
    $this->strategy = new AdvancedBypassStrategy([], 'advanced_bypass', []);
  }

  public function testVipUserBypass(): void {
    $account = $this->createMock(AccountInterface::class);
    $account->method('hasRole')->with('vip_user')->willReturn(TRUE);

    $route_match = $this->createMock(RouteMatchInterface::class);

    $should_bypass = $this->strategy->shouldBypass('Test message', 'status', $account, $route_match);

    $this->assertTrue($should_bypass);
  }

  public function testEmergencyMessageBypass(): void {
    $account = $this->createMock(AccountInterface::class);
    $account->method('hasRole')->willReturn(FALSE);

    $route_match = $this->createMock(RouteMatchInterface::class);

    $should_bypass = $this->strategy->shouldBypass('URGENT: System failure', 'error', $account, $route_match);

    $this->assertTrue($should_bypass);
  }
}
```

### Performance Considerations

```php
/**
 * Optimisations pour de gros volumes :
 *
 * 1. Cache les règles par utilisateur
 * 2. Évitez les appels DB répétés
 * 3. Utilisez le cache statique
 */
class OptimizedFilteredMessenger extends FilteredMessenger {

  private array $cachedRules = [];
  private array $cachedBypassChecks = [];

  protected function shouldBlockMessage($message, $type) {
    // Cache key based on user roles + message hash
    $cache_key = md5($this->currentUser->id() . serialize($this->currentUser->getRoles()) . $message . $type);

    if (isset($this->cachedBypassChecks[$cache_key])) {
      return $this->cachedBypassChecks[$cache_key];
    }

    $result = parent::shouldBlockMessage($message, $type);
    $this->cachedBypassChecks[$cache_key] = $result;

    return $result;
  }
}
```

### API du service UnfilteredMessenger

```php
// Ajouter un message non filtré
$service->addMessage($message, $type, $context, $display);

// Vérifier si un message est non filtré
$is_unfiltered = $service->isUnfilteredMessage($message, $type);

// Obtenir tous les messages non filtrés
$messages = $service->getUnfilteredMessages();

// Supprimer un message spécifique
$service->removeMessage($message, $type);

// Vider tous les messages
$service->clearUnfilteredMessages();
```

## Tests

Le module inclut une suite complète de tests unitaires compatible avec l'infrastructure drupal.org :

```bash
# Depuis la racine du projet Drupal
vendor/bin/phpunit --bootstrap=web/core/tests/bootstrap.php \
  web/modules/custom/message_filter/tests/src/Unit/

# Ou avec la configuration Drupal Core
vendor/bin/phpunit -c web/core/phpunit.xml.dist \
  web/modules/custom/message_filter/tests/src/Unit/
```

### Configuration pour développeurs

Créez votre propre `phpunit.xml` si nécessaire (non inclus dans le module) :

```xml
<?xml version="1.0" encoding="UTF-8"?>
<phpunit bootstrap="web/core/tests/bootstrap.php" colors="true">
  <testsuites>
    <testsuite name="message_filter">
      <directory>web/modules/custom/message_filter/tests/src/Unit</directory>
    </testsuite>
  </testsuites>
</phpunit>
```

### Couverture des tests

- **29 tests**, **71 assertions**
- Service principal : UnfilteredMessengerService
- Plugin système : DemoBypassStrategy
- Formulaire : MessageFilterSettingsForm
- Tests unitaires isolés sans dépendances container

## Ordre de priorité

Le système applique les règles dans cet ordre :

1. **Messages non filtrés** : Jamais bloqués (priorité absolue)
2. **Permission bypass** : bypass message filter
3. **Règles par rôle** : Selon priorité configurée
4. **Plugins de contournement** : Selon weight des plugins
5. **Configuration globale** : Si aucune règle spécifique

## Structure des fichiers

```text
message_filter/
├── config/
│   ├── install/message_filter.settings.yml
│   └── schema/message_filter.schema.yml
├── src/
│   ├── Annotation/MessageBypassStrategy.php
│   ├── Form/MessageFilterSettingsForm.php
│   ├── Messenger/FilteredMessenger.php
│   ├── Plugin/
│   │   ├── MessageBypassStrategy/
│   │   │   ├── DemoBypassStrategy.php
│   │   │   └── ExampleBypassStrategy.php
│   │   ├── MessageBypassStrategyInterface.php
│   │   └── MessageBypassStrategyManager.php
│   └── Service/UnfilteredMessengerService.php
├── tests/src/Unit/
│   ├── Form/MessageFilterSettingsFormSimpleTest.php
│   ├── Plugin/MessageBypassStrategy/DemoBypassStrategySimpleTest.php
│   └── Service/
│       ├── UnfilteredMessengerServiceTest.php
│       └── UnfilteredMessengerServiceSimpleTest.php
├── message_filter.info.yml
├── message_filter.links.menu.yml
├── message_filter.module
├── message_filter.permissions.yml
├── message_filter.routing.yml
└── message_filter.services.yml
```

## Contribution

### Standards de développement

- **Drupal Coding Standards** : PSR-12 + Drupal specifics
- **Documentation** : PHPDoc complet
- **Tests** : Couverture > 80% requise
- **Types stricts** : declare(strict_types=1)

## Licence

GPL-2.0+ - Compatible avec Drupal Core

---

**Développé pour Drupal 10/11** | **Architecture moderne**
