<?php

namespace Drupal\message_filter\Service;

use Psr\Log\LoggerInterface;

/**
 * Service for managing unfiltered messages that should not be filtered.
 */
class UnfilteredMessengerService {

  /**
   * Array of unfiltered messages.
   *
   * @var array
   */
  protected $unfilteredMessages = [];

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * Constructs an UnfilteredMessengerService object.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   */
  public function __construct(LoggerInterface $logger) {
    $this->logger = $logger;
  }

  /**
   * Gets the messenger service.
   *
   * @return \Drupal\Core\Messenger\MessengerInterface
   *   The messenger service.
   */
  protected function getMessenger() {
    return \Drupal::messenger();
  }

  /**
   * Adds an unfiltered message that should not be filtered.
   *
   * @param mixed $message
   *   The message to add.
   * @param string $type
   *   The message type.
   * @param array $context
   *   Additional context for the message.
   * @param bool $display
   *   Whether to display the message immediately.
   *
   * @return $this
   */
  public function addMessage($message, $type = 'status', array $context = [], $display = TRUE) {
    // Toujours marquer comme unfiltered d'abord
    $message_hash = $this->generateMessageHash($message, $type);
    $this->unfilteredMessages[$message_hash] = [
      'message' => $message,
      'type' => $type,
      'context' => $context,
      'timestamp' => time(),
    ];

    $this->logger->debug('Unfiltered message added: @message (@type)', [
      '@message' => is_string($message) ? $message : serialize($message),
      '@type' => $type,
    ]);

    // Afficher immédiatement le message si demandé
    if ($display) {
      $this->displayMessage($message, $type);
    }

    return $this;
  }

  /**
   * Displays a message using Drupal's messenger service.
   *
   * @param mixed $message
   *   The message to display.
   * @param string $type
   *   The message type.
   */
  protected function displayMessage($message, $type) {
    // Le message est déjà marqué comme unfiltered dans addMessage()
    // Donc FilteredMessenger pourra le détecter avec isUnfilteredMessage()

    $messenger = $this->getMessenger();

    switch ($type) {
      case 'status':
        $messenger->addStatus($message);
        break;
      case 'warning':
        $messenger->addWarning($message);
        break;
      case 'error':
        $messenger->addError($message);
        break;
      default:
        $messenger->addMessage($message, $type);
        break;
    }
  }  /**
   * Adds and displays an unfiltered status message.
   *
   * @param mixed $message
   *   The message to add.
   * @param array $context
   *   Additional context for the message.
   *
   * @return $this
   */
  public function addStatus($message, array $context = []) {
    return $this->addMessage($message, 'status', $context);
  }

  /**
   * Adds and displays an unfiltered warning message.
   *
   * @param mixed $message
   *   The message to add.
   * @param array $context
   *   Additional context for the message.
   *
   * @return $this
   */
  public function addWarning($message, array $context = []) {
    return $this->addMessage($message, 'warning', $context);
  }

  /**
   * Adds and displays an unfiltered error message.
   *
   * @param mixed $message
   *   The message to add.
   * @param array $context
   *   Additional context for the message.
   *
   * @return $this
   */
  public function addError($message, array $context = []) {
    return $this->addMessage($message, 'error', $context);
  }

  /**
   * Checks if a message is an unfiltered message.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if the message is an unfiltered message, FALSE otherwise.
   */
  public function isUnfilteredMessage($message, $type) {
    $message_hash = $this->generateMessageHash($message, $type);
    return isset($this->unfilteredMessages[$message_hash]);
  }

  /**
   * Alias for isUnfilteredMessage for backward compatibility.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if the message is an unfiltered message, FALSE otherwise.
   */
  public function isCustomMessage($message, $type) {
    return $this->isUnfilteredMessage($message, $type);
  }

  /**
   * Removes an unfiltered message.
   *
   * @param mixed $message
   *   The message to remove.
   * @param string $type
   *   The message type.
   *
   * @return $this
   */
  public function removeMessage($message, $type) {
    $message_hash = $this->generateMessageHash($message, $type);
    unset($this->unfilteredMessages[$message_hash]);
    return $this;
  }

  /**
   * Gets all unfiltered messages.
   *
   * @return array
   *   Array of unfiltered messages.
   */
  public function getUnfilteredMessages() {
    return $this->unfilteredMessages;
  }

  /**
   * Alias for getUnfilteredMessages for backward compatibility.
   *
   * @return array
   *   Array of unfiltered messages.
   */
  public function getCustomMessages() {
    return $this->getUnfilteredMessages();
  }

  /**
   * Clears all unfiltered messages.
   *
   * @return $this
   */
  public function clearUnfilteredMessages() {
    $this->unfilteredMessages = [];
    return $this;
  }

  /**
   * Alias for clearUnfilteredMessages for backward compatibility.
   *
   * @return $this
   */
  public function clearCustomMessages() {
    return $this->clearUnfilteredMessages();
  }

  /**
   * Generates a hash for a message.
   *
   * @param mixed $message
   *   The message.
   * @param string $type
   *   The message type.
   *
   * @return string
   *   The message hash.
   */
  protected function generateMessageHash($message, $type) {
    $message_string = is_string($message) ? $message : serialize($message);
    return md5($message_string . $type);
  }

}
