<?php

declare(strict_types=1);

namespace Drupal\message_filter\Service;

use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Service for evaluating message filtering rules.
 */
class MessageFilterRuleEvaluator implements MessageFilterRuleEvaluatorInterface {

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The URL matcher service.
   *
   * @var \Drupal\message_filter\Service\UrlMatcherInterface
   */
  protected UrlMatcherInterface $urlMatcher;

  /**
   * Constructs a MessageFilterRuleEvaluator object.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\message_filter\Service\UrlMatcherInterface $url_matcher
   *   The URL matcher service.
   */
  public function __construct(
    RouteMatchInterface $route_match,
    RequestStack $request_stack,
    UrlMatcherInterface $url_matcher,
  ) {
    $this->routeMatch = $route_match;
    $this->requestStack = $request_stack;
    $this->urlMatcher = $url_matcher;
  }

  /**
   * {@inheritdoc}
   */
  public function shouldBlockMessage($message, string $type, AccountInterface $account, array $rules): bool {
    if (empty($rules)) {
      return FALSE;
    }

    $current_route = $this->routeMatch->getRouteName();
    $current_path = $this->getCurrentPath();

    foreach ($rules as $rule) {
      if ($this->evaluateRule($rule, $type, $current_route, $current_path)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Evaluates a single rule.
   *
   * @param array $rule
   *   The rule to evaluate.
   * @param string $message_type
   *   The message type.
   * @param string|null $current_route
   *   The current route name.
   * @param string $current_path
   *   The current path.
   *
   * @return bool
   *   TRUE if the rule applies and message should be blocked.
   */
  protected function evaluateRule(array $rule, string $message_type, ?string $current_route, string $current_path): bool {
    // If "block all messages" is enabled, block everything.
    if (!empty($rule['block_all_messages'])) {
      return TRUE;
    }

    $blocked_routes = $rule['blocked_routes'] ?? [];
    $blocked_urls = $rule['blocked_urls'] ?? [];
    $blocked_types = $rule['blocked_message_types'] ?? [];

    // Check if current context matches blocked routes/URLs.
    $context_matches = $this->contextMatches($current_route, $current_path, $blocked_routes, $blocked_urls);

    // If specific routes/URLs are defined, only filter when they match.
    if (!empty($blocked_routes) || !empty($blocked_urls)) {
      if ($context_matches && $this->messageTypeMatches($message_type, $blocked_types)) {
        return TRUE;
      }
    }
    else {
      // No specific routes defined, apply message type filtering globally.
      if ($this->messageTypeMatches($message_type, $blocked_types)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Checks if current context matches blocked routes or URLs.
   *
   * @param string|null $current_route
   *   The current route name.
   * @param string $current_path
   *   The current path.
   * @param array $blocked_routes
   *   Array of blocked routes.
   * @param array $blocked_urls
   *   Array of blocked URL patterns.
   *
   * @return bool
   *   TRUE if context matches.
   */
  protected function contextMatches(?string $current_route, string $current_path, array $blocked_routes, array $blocked_urls): bool {
    // Check blocked routes.
    if (!empty($blocked_routes) && $current_route && in_array($current_route, $blocked_routes, TRUE)) {
      return TRUE;
    }

    // Check blocked URLs.
    if (!empty($blocked_urls) && $this->urlMatcher->matchesAny($current_path, $blocked_urls)) {
      return TRUE;
    }

    return FALSE;
  }

  /**
   * Checks if message type should be blocked.
   *
   * @param string $message_type
   *   The message type.
   * @param array $blocked_types
   *   Array of blocked message types.
   *
   * @return bool
   *   TRUE if message type should be blocked.
   */
  protected function messageTypeMatches(string $message_type, array $blocked_types): bool {
    return !empty($blocked_types) && in_array($message_type, $blocked_types, TRUE);
  }

  /**
   * Gets the current path from the request.
   *
   * @return string
   *   The current path.
   */
  protected function getCurrentPath(): string {
    $request = $this->requestStack->getCurrentRequest();
    return $request ? $request->getPathInfo() : '';
  }

}
