<?php

declare(strict_types=1);

namespace Drupal\message_filter\Service;

use Psr\Log\LoggerInterface;
use Drupal\message_filter\Factory\MessengerFactoryInterface;

/**
 * Service for managing unfiltered messages that should not be filtered.
 */
class UnfilteredMessengerService implements UnfilteredMessengerServiceInterface {

  /**
   * Array of unfiltered messages.
   *
   * @var array
   */
  protected array $unfilteredMessages = [];

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The messenger factory.
   *
   * @var \Drupal\message_filter\Factory\MessengerFactoryInterface
   */
  protected MessengerFactoryInterface $messengerFactory;

  /**
   * Constructs an UnfilteredMessengerService object.
   *
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\message_filter\Factory\MessengerFactoryInterface $messenger_factory
   *   The messenger factory.
   */
  public function __construct(LoggerInterface $logger, MessengerFactoryInterface $messenger_factory) {
    $this->logger = $logger;
    $this->messengerFactory = $messenger_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function addMessage($message, string $type = 'status', array $context = [], bool $display = TRUE): self {
    // Always mark as unfiltered first.
    $message_hash = $this->generateMessageHash($message, $type);
    $this->unfilteredMessages[$message_hash] = [
      'message' => $message,
      'type' => $type,
      'context' => $context,
      'timestamp' => time(),
    ];

    $this->logger->debug('Unfiltered message added: @message (@type)', [
      '@message' => is_string($message) ? $message : serialize($message),
      '@type' => $type,
    ]);

    // Display message immediately if requested.
    if ($display) {
      $this->displayMessage($message, $type);
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function addStatus($message, array $context = []): self {
    return $this->addMessage($message, 'status', $context);
  }

  /**
   * {@inheritdoc}
   */
  public function addWarning($message, array $context = []): self {
    return $this->addMessage($message, 'warning', $context);
  }

  /**
   * {@inheritdoc}
   */
  public function addError($message, array $context = []): self {
    return $this->addMessage($message, 'error', $context);
  }

  /**
   * {@inheritdoc}
   */
  public function isUnfilteredMessage($message, string $type): bool {
    $message_hash = $this->generateMessageHash($message, $type);
    return isset($this->unfilteredMessages[$message_hash]);
  }

  /**
   * {@inheritdoc}
   */
  public function removeMessage($message, string $type): self {
    $message_hash = $this->generateMessageHash($message, $type);
    unset($this->unfilteredMessages[$message_hash]);
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getUnfilteredMessages(): array {
    return $this->unfilteredMessages;
  }

  /**
   * {@inheritdoc}
   */
  public function clearUnfilteredMessages(): self {
    $this->unfilteredMessages = [];
    return $this;
  }

  /**
   * Backward compatibility aliases.
   */

  /**
   * Alias for isUnfilteredMessage for backward compatibility.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if the message is an unfiltered message.
   */
  public function isCustomMessage($message, string $type): bool {
    return $this->isUnfilteredMessage($message, $type);
  }

  /**
   * Alias for getUnfilteredMessages for backward compatibility.
   *
   * @return array
   *   Array of unfiltered messages.
   */
  public function getCustomMessages(): array {
    return $this->getUnfilteredMessages();
  }

  /**
   * Alias for clearUnfilteredMessages for backward compatibility.
   *
   * @return $this
   */
  public function clearCustomMessages(): self {
    return $this->clearUnfilteredMessages();
  }

  /**
   * Displays a message using the original messenger service.
   *
   * @param mixed $message
   *   The message to display.
   * @param string $type
   *   The message type.
   */
  protected function displayMessage($message, string $type): void {
    // The message is already marked as unfiltered in addMessage()
    // So FilteredMessenger can detect it with isUnfilteredMessage()
    // Use factory to get original messenger and avoid circular dependency.
    $messenger = $this->messengerFactory->getMessenger();
    switch ($type) {
      case 'status':
        $messenger->addStatus($message);
        break;

      case 'warning':
        $messenger->addWarning($message);
        break;

      case 'error':
        $messenger->addError($message);
        break;

      default:
        $messenger->addMessage($message, $type);
        break;
    }
  }

  /**
   * Generates a hash for a message.
   *
   * @param mixed $message
   *   The message.
   * @param string $type
   *   The message type.
   *
   * @return string
   *   The message hash.
   */
  protected function generateMessageHash($message, string $type): string {
    $message_string = is_string($message) ? $message : serialize($message);
    return md5($message_string . $type);
  }

}
