<?php

declare(strict_types=1);

namespace Drupal\message_filter\Service;

/**
 * Service for URL pattern matching with wildcard support.
 */
class UrlMatcher implements UrlMatcherInterface {

  /**
   * {@inheritdoc}
   */
  public function matches(string $path, string $pattern): bool {
    // Exact match.
    if ($path === $pattern) {
      return TRUE;
    }

    // Wildcard pattern matching.
    if (strpos($pattern, '*') !== FALSE) {
      return $this->matchesWildcard($path, $pattern);
    }

    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function matchesAny(string $path, array $patterns): bool {
    foreach ($patterns as $pattern) {
      if ($this->matches($path, $pattern)) {
        return TRUE;
      }
    }

    return FALSE;
  }

  /**
   * Matches a path against a wildcard pattern.
   *
   * @param string $path
   *   The path to match.
   * @param string $pattern
   *   The wildcard pattern.
   *
   * @return bool
   *   TRUE if the path matches the pattern.
   */
  protected function matchesWildcard(string $path, string $pattern): bool {
    // Escape special regex characters except *.
    $escaped_pattern = preg_quote($pattern, '/');

    // Replace escaped \* with .*.
    $regex_pattern = str_replace('\*', '.*', $escaped_pattern);

    // Add anchors for full match.
    $regex_pattern = '/^' . $regex_pattern . '$/';

    return preg_match($regex_pattern, $path) === 1;
  }

}
