<?php

declare(strict_types=1);

namespace Drupal\Tests\message_filter\Unit\Messenger;

use Drupal\message_filter\Messenger\FilteredMessenger;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\message_filter\Service\MessageFilterConfigurationInterface;
use Drupal\message_filter\Service\MessageFilterRuleEvaluatorInterface;
use Drupal\message_filter\Service\UnfilteredMessengerServiceInterface;
use Drupal\message_filter\Plugin\MessageBypassStrategyManager;
use Drupal\Core\Routing\RouteMatchInterface;
use PHPUnit\Framework\TestCase;

/**
 * Tests for the FilteredMessenger class.
 */
class FilteredMessengerTest extends TestCase {

  /**
   * The messenger service used for sending messages.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  private $messenger;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  private $currentUser;

  /**
   * The configuration array for the messenger.
   *
   * @var array
   */
  private $configuration;
  /**
   * The rule evaluator service for message filtering.
   *
   * @var \Drupal\message_filter\RuleEvaluatorInterface
   */
  private $ruleEvaluator;

  /**
   * The unfiltered messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  private $unfilteredMessenger;
  /**
   * The bypass strategy manager service.
   *
   * @var \Drupal\Component\Plugin\PluginManagerInterface
   */
  private $bypassStrategyManager;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  private $routeMatch;

  /**
   * The filtered messenger instance under test.
   *
   * @var \Drupal\message_filter\Messenger\FilteredMessenger
   */
  private $filteredMessenger;

  /**
   * Sets up the test.
   */
  protected function setUp(): void {
    parent::setUp();
    $this->messenger = $this->createMock(MessengerInterface::class);
    $this->currentUser = $this->createMock(AccountInterface::class);
    $this->configuration = $this->createMock(MessageFilterConfigurationInterface::class);
    $this->ruleEvaluator = $this->createMock(MessageFilterRuleEvaluatorInterface::class);
    $this->unfilteredMessenger = $this->createMock(UnfilteredMessengerServiceInterface::class);
    $this->bypassStrategyManager = $this->createMock(MessageBypassStrategyManager::class);
    $this->routeMatch = $this->createMock(RouteMatchInterface::class);
    $this->filteredMessenger = new FilteredMessenger(
      $this->messenger,
      $this->currentUser,
      $this->configuration,
      $this->ruleEvaluator,
      $this->unfilteredMessenger,
      $this->bypassStrategyManager,
      $this->routeMatch
    );
  }

  /**
   * Tests adding a message that is not blocked.
   */
  public function testAddMessageNotBlocked() {
    $this->configuration->method('isFilteringEnabled')->willReturn(TRUE);
    $this->unfilteredMessenger->method('isUnfilteredMessage')->willReturn(FALSE);
    $this->currentUser->method('hasPermission')->willReturn(FALSE);
    $this->configuration->method('getRoleRulesForUser')->willReturn([]);
    $this->ruleEvaluator->method('shouldBlockMessage')->willReturn(FALSE);
    $this->bypassStrategyManager->method('getDefinitions')->willReturn([]);
    $this->messenger->expects($this->once())->method('addMessage')->with('foo', 'status', FALSE);
    $this->filteredMessenger->addMessage('foo', 'status', FALSE);
  }

  /**
   * Tests adding a message that is blocked by a filtering rule.
   */
  public function testAddMessageBlockedByRule() {
    $this->configuration->method('isFilteringEnabled')->willReturn(TRUE);
    $this->unfilteredMessenger->method('isUnfilteredMessage')->willReturn(FALSE);
    $this->currentUser->method('hasPermission')->willReturn(FALSE);
    $this->configuration->method('getRoleRulesForUser')->willReturn([]);
    $this->ruleEvaluator->method('shouldBlockMessage')->willReturn(TRUE);
    $this->bypassStrategyManager->method('getDefinitions')->willReturn([]);
    $this->messenger->expects($this->never())->method('addMessage');
    $this->filteredMessenger->addMessage('foo', 'status', FALSE);
  }

  /**
   * Tests adding an unfiltered message.
   */
  public function testAddMessageUnfiltered() {
    $this->configuration->method('isFilteringEnabled')->willReturn(TRUE);
    $this->unfilteredMessenger->method('isUnfilteredMessage')->willReturn(TRUE);
    $this->messenger->expects($this->once())->method('addMessage')->with('foo', 'status', FALSE);
    $this->filteredMessenger->addMessage('foo', 'status', FALSE);
  }

  /**
   * Tests adding a message with bypass permission.
   */
  public function testAddMessageBypassPermission() {
    $this->configuration->method('isFilteringEnabled')->willReturn(TRUE);
    $this->unfilteredMessenger->method('isUnfilteredMessage')->willReturn(FALSE);
    $this->currentUser->method('hasPermission')->willReturn(TRUE);
    $this->messenger->expects($this->once())->method('addMessage')->with('foo', 'status', FALSE);
    $this->filteredMessenger->addMessage('foo', 'status', FALSE);
  }

}
