<?php

declare(strict_types=1);

namespace Drupal\message_filter\Factory;

use Drupal\Core\Messenger\MessengerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Factory for creating messenger instances.
 *
 * This factory helps avoid circular dependencies when the
 * UnfilteredMessengerService needs to access the actual messenger service
 * (which is decorated by FilteredMessenger).
 *
 * Instead of injecting MessengerInterface directly (which would create a
 * circular dependency), we inject this factory that can lazily resolve the
 * original messenger.
 */
class MessengerFactory implements MessengerFactoryInterface {

  /**
   * The service container.
   *
   * @var \Symfony\Component\DependencyInjection\ContainerInterface
   */
  protected ContainerInterface $container;

  /**
   * Constructs a MessengerFactory object.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The service container.
   */
  public function __construct(ContainerInterface $container) {
    $this->container = $container;
  }

  /**
   * {@inheritdoc}
   */
  public function getMessenger(): MessengerInterface {
    // Get the original messenger service (before decoration)
    // This avoids the circular dependency with FilteredMessenger.
    return $this->container->get('messenger.original');
  }

}
