<?php

declare(strict_types=1);

namespace Drupal\message_filter\Messenger;

use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\message_filter\Plugin\MessageBypassStrategyManager;
use Drupal\message_filter\Service\MessageFilterConfigurationInterface;
use Drupal\message_filter\Service\MessageFilterRuleEvaluatorInterface;
use Drupal\message_filter\Service\UnfilteredMessengerServiceInterface;

/**
 * Decorator for MessengerInterface that filters messages based on config.
 *
 * This class follows SOLID principles:
 * - SRP: Single responsibility - only handles message filtering delegation
 * - OCP: Open for extension via interfaces
 * - LSP: Substitutable for MessengerInterface
 * - ISP: Uses specific interfaces for each dependency
 * - DIP: Depends on abstractions, not concretions.
 */
class FilteredMessenger implements MessengerInterface {

  /**
   * The decorated messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface
   */
  protected MessengerInterface $messenger;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The configuration service.
   *
   * @var \Drupal\message_filter\Service\MessageFilterConfigurationInterface
   */
  protected MessageFilterConfigurationInterface $configuration;

  /**
   * The rule evaluator service.
   *
   * @var \Drupal\message_filter\Service\MessageFilterRuleEvaluatorInterface
   */
  protected MessageFilterRuleEvaluatorInterface $ruleEvaluator;

  /**
   * The unfiltered messenger service.
   *
   * @var \Drupal\message_filter\Service\UnfilteredMessengerServiceInterface
   */
  protected UnfilteredMessengerServiceInterface $unfilteredMessenger;

  /**
   * The bypass strategy manager.
   *
   * @var \Drupal\message_filter\Plugin\MessageBypassStrategyManager
   */
  protected MessageBypassStrategyManager $bypassStrategyManager;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Constructs a FilteredMessenger object.
   *
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The decorated messenger service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\message_filter\Service\MessageFilterConfigurationInterface $configuration
   *   The configuration service.
   * @param \Drupal\message_filter\Service\MessageFilterRuleEvaluatorInterface $rule_evaluator
   *   The rule evaluator service.
   * @param \Drupal\message_filter\Service\UnfilteredMessengerServiceInterface $unfiltered_messenger
   *   The unfiltered messenger service.
   * @param \Drupal\message_filter\Plugin\MessageBypassStrategyManager $bypass_strategy_manager
   *   The bypass strategy manager.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   */
  public function __construct(
    MessengerInterface $messenger,
    AccountInterface $current_user,
    MessageFilterConfigurationInterface $configuration,
    MessageFilterRuleEvaluatorInterface $rule_evaluator,
    UnfilteredMessengerServiceInterface $unfiltered_messenger,
    MessageBypassStrategyManager $bypass_strategy_manager,
    RouteMatchInterface $route_match,
  ) {
    $this->messenger = $messenger;
    $this->currentUser = $current_user;
    $this->configuration = $configuration;
    $this->ruleEvaluator = $rule_evaluator;
    $this->unfilteredMessenger = $unfiltered_messenger;
    $this->bypassStrategyManager = $bypass_strategy_manager;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public function addMessage(
    $message,
    $type = self::TYPE_STATUS,
    $repeat = FALSE,
  ) {
    if (!$this->shouldBlockMessage($message, $type)) {
      return $this->messenger->addMessage($message, $type, $repeat);
    }

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function addStatus($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_STATUS, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addError($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_ERROR, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function addWarning($message, $repeat = FALSE) {
    return $this->addMessage($message, static::TYPE_WARNING, $repeat);
  }

  /**
   * {@inheritdoc}
   */
  public function all() {
    return $this->messenger->all();
  }

  /**
   * {@inheritdoc}
   */
  public function messagesByType($type) {
    return $this->messenger->messagesByType($type);
  }

  /**
   * {@inheritdoc}
   */
  public function deleteAll() {
    return $this->messenger->deleteAll();
  }

  /**
   * {@inheritdoc}
   */
  public function deleteByType($type) {
    return $this->messenger->deleteByType($type);
  }

  /**
   * Determines if a message should be blocked using a chain of responsibility.
   *
   * Priority order:
   * 1. Unfiltered messages (NEVER blocked)
   * 2. Bypass permission
   * 3. Role-based rules
   * 4. Bypass strategy plugins.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if the message should be blocked.
   */
  protected function shouldBlockMessage($message, string $type): bool {
    // If filtering is disabled globally, don't block anything.
    if (!$this->configuration->isFilteringEnabled()) {
      return FALSE;
    }

    // PRIORITY 1: Unfiltered messages - NEVER block these.
    if ($this->unfilteredMessenger->isUnfilteredMessage($message, $type)) {
      return FALSE;
    }

    // PRIORITY 2: Check bypass permission.
    if ($this->currentUser->hasPermission('bypass message filter')) {
      return FALSE;
    }

    // PRIORITY 3: Apply role-based rules.
    $role_rules = $this->configuration->getRoleRulesForUser($this->currentUser);
    if ($this->ruleEvaluator->shouldBlockMessage(
      $message,
      $type,
      $this->currentUser,
      $role_rules
    )) {
      return TRUE;
    }

    // PRIORITY 4: Check bypass strategy plugins.
    if ($this->shouldBypassViaPlugins($message, $type)) {
      return FALSE;
    }

    return FALSE;
  }

  /**
   * Checks if message should bypass filtering via strategy plugins.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if message should bypass filtering.
   */
  protected function shouldBypassViaPlugins($message, string $type): bool {
    foreach ($this->bypassStrategyManager->getDefinitions() as $plugin_id => $definition) {
      // Suppress unused variable warning.
      unset($definition);
      $strategy = $this->bypassStrategyManager->createInstance($plugin_id);
      if ($strategy->shouldBypass(
        $message,
        $type,
        $this->currentUser,
        $this->routeMatch
      )) {
        return TRUE;
      }
    }

    return FALSE;
  }

}
