<?php

declare(strict_types=1);

namespace Drupal\message_filter\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;

/**
 * Service for managing message filter configuration.
 */
class MessageFilterConfiguration implements MessageFilterConfigurationInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * Constructs a MessageFilterConfiguration object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(ConfigFactoryInterface $config_factory) {
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public function isFilteringEnabled(): bool {
    return (bool) $this->configFactory
      ->get('message_filter.settings')
      ->get('enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function getRoleRulesForUser(AccountInterface $account): array {
    $config = $this->configFactory->get('message_filter.settings');
    $role_rules = $config->get('role_rules') ?: [];
    $user_roles = $account->getRoles();

    $applicable_rules = [];
    foreach ($user_roles as $role_id) {
      if (isset($role_rules[$role_id]) && !empty($role_rules[$role_id]['enabled'])) {
        $applicable_rules[] = $role_rules[$role_id];
      }
    }

    // Sort by priority (highest first).
    usort($applicable_rules, function (array $rule_a, array $rule_b): int {
      return ($rule_b['priority'] ?? 0) <=> ($rule_a['priority'] ?? 0);
    });

    return $applicable_rules;
  }

  /**
   * {@inheritdoc}
   */
  public function getGlobalConfiguration(): array {
    return $this->configFactory
      ->get('message_filter.settings')
      ->get() ?: [];
  }

}
