<?php

namespace Drupal\message_filter\Service;

use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Service for validating message filter form data.
 */
class MessageFilterFormValidator {

  use StringTranslationTrait;

  /**
   * Validates role rules from form state.
   *
   * @param array $role_rules_values
   *   The role rules values from form state.
   *
   * @return array
   *   Array of validation errors keyed by field path.
   */
  public function validateRoleRules(array $role_rules_values): array {
    $errors = [];

    foreach ($role_rules_values as $role_id => $role_data) {
      if (!empty($role_data['enabled'])) {
        // Validate routes format.
        if (!empty($role_data['blocked_routes'])) {
          $route_errors = $this->validateRoutes($role_data['blocked_routes'], $role_id);
          $errors = array_merge($errors, $route_errors);
        }

        // Validate URLs format.
        if (!empty($role_data['blocked_urls'])) {
          $url_errors = $this->validateUrls($role_data['blocked_urls'], $role_id);
          $errors = array_merge($errors, $url_errors);
        }

        // Validate priority range.
        $priority_errors = $this->validatePriority($role_data['priority'] ?? 0, $role_id);
        $errors = array_merge($errors, $priority_errors);
      }
    }

    return $errors;
  }

  /**
   * Validates route names.
   *
   * @param string $routes_string
   *   The routes string from form.
   * @param string $role_id
   *   The role ID for error context.
   *
   * @return array
   *   Array of validation errors.
   */
  protected function validateRoutes(string $routes_string, string $role_id): array {
    $errors = [];
    $routes = explode("\n", $routes_string);

    foreach ($routes as $line_number => $route) {
      $route = trim($route);
      if (!empty($route)) {
        // Basic validation: route names should not contain spaces or special characters except dots and underscores.
        if (!preg_match('/^[a-zA-Z0-9_.]+$/', $route)) {
          $errors["role_rules][{$role_id}][blocked_routes"] = $this->t('Invalid route name "@route" on line @line. Route names should only contain letters, numbers, dots, and underscores.', [
            '@route' => $route,
            '@line' => $line_number + 1,
          ]);
        }
      }
    }

    return $errors;
  }

  /**
   * Validates URL patterns.
   *
   * @param string $urls_string
   *   The URLs string from form.
   * @param string $role_id
   *   The role ID for error context.
   *
   * @return array
   *   Array of validation errors.
   */
  protected function validateUrls(string $urls_string, string $role_id): array {
    $errors = [];
    $urls = explode("\n", $urls_string);

    foreach ($urls as $line_number => $url) {
      $url = trim($url);
      if (!empty($url)) {
        // Basic validation: URLs should start with / and contain valid path characters
        if (!preg_match('/^\/[a-zA-Z0-9\/_\-\*\.]*$/', $url)) {
          $errors["role_rules][{$role_id}][blocked_urls"] = $this->t('Invalid URL path "@url" on line @line. URLs should start with "/" and contain only valid path characters (letters, numbers, /, -, _, *, .).', [
            '@url' => $url,
            '@line' => $line_number + 1,
          ]);
        }
      }
    }

    return $errors;
  }

  /**
   * Validates priority value.
   *
   * @param int $priority
   *   The priority value.
   * @param string $role_id
   *   The role ID for error context.
   *
   * @return array
   *   Array of validation errors.
   */
  protected function validatePriority(int $priority, string $role_id): array {
    $errors = [];

    if ($priority < 0 || $priority > 100) {
      $errors["role_rules][{$role_id}][priority"] = $this->t('Priority must be between 0 and 100.');
    }

    return $errors;
  }

}
