<?php

namespace Drupal\message_filter\Service;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\message_filter\Plugin\MessageBypassStrategyManager;

/**
 * Service for handling message filtering logic.
 */
class MessageFilterService implements MessageFilterServiceInterface {

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * The unfiltered messenger service.
   *
   * @var \Drupal\message_filter\Service\UnfilteredMessengerService
   */
  protected $unfilteredMessenger;

  /**
   * The bypass strategy manager.
   *
   * @var \Drupal\message_filter\Plugin\MessageBypassStrategyManager
   */
  protected $bypassStrategyManager;

  /**
   * The URL matcher service.
   *
   * @var \Drupal\message_filter\Service\UrlMatcherServiceInterface
   */
  protected $urlMatcher;

  /**
   * Constructs a MessageFilterService object.
   *
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The current route match.
   * @param \Drupal\message_filter\Service\UnfilteredMessengerService $unfiltered_messenger
   *   The unfiltered messenger service.
   * @param \Drupal\message_filter\Plugin\MessageBypassStrategyManager $bypass_strategy_manager
   *   The bypass strategy manager.
   * @param \Drupal\message_filter\Service\UrlMatcherServiceInterface $url_matcher
   *   The URL matcher service.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    AccountInterface $current_user,
    RouteMatchInterface $route_match,
    UnfilteredMessengerService $unfiltered_messenger,
    MessageBypassStrategyManager $bypass_strategy_manager,
    UrlMatcherServiceInterface $url_matcher
  ) {
    $this->configFactory = $config_factory;
    $this->currentUser = $current_user;
    $this->routeMatch = $route_match;
    $this->unfilteredMessenger = $unfiltered_messenger;
    $this->bypassStrategyManager = $bypass_strategy_manager;
    $this->urlMatcher = $url_matcher;
  }

  /**
   * {@inheritdoc}
   */
  public function isFilteringEnabled(): bool {
    return (bool) $this->configFactory->get('message_filter.settings')->get('enabled');
  }

  /**
   * {@inheritdoc}
   */
  public function shouldBlockMessage($message, string $type): bool {
    // Short-circuit if filtering is disabled.
    if (!$this->isFilteringEnabled()) {
      return FALSE;
    }

    // Never block unfiltered messages - they have absolute priority.
    if ($this->unfilteredMessenger->isUnfilteredMessage($message, $type)) {
      return FALSE;
    }

    // Check if user has bypass permission.
    if ($this->currentUser->hasPermission('bypass message filtering')) {
      return FALSE;
    }

    // Get role-based rules and check them.
    $config = $this->configFactory->get('message_filter.settings');
    $role_rules = $config->get('role_rules') ?: [];

    return $this->shouldBlockByRoleRules($message, $type, $role_rules);
  }

  /**
   * Checks if message should be blocked based on role rules.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   * @param array $role_rules
   *   The role rules configuration.
   *
   * @return bool
   *   TRUE if the message should be blocked, FALSE otherwise.
   */
  protected function shouldBlockByRoleRules($message, string $type, array $role_rules): bool {
    $user_roles = $this->currentUser->getRoles();
    $applicable_rules = [];

    // Collect rules for user's roles.
    foreach ($user_roles as $role_id) {
      if (isset($role_rules[$role_id]) && !empty($role_rules[$role_id]['enabled'])) {
        $applicable_rules[] = $role_rules[$role_id];
      }
    }

    if (empty($applicable_rules)) {
      return $this->checkBypassStrategies($message, $type);
    }

    // Sort by priority (higher priority first).
    usort($applicable_rules, function ($a, $b) {
      return ($b['priority'] ?? 0) <=> ($a['priority'] ?? 0);
    });

    // Apply the highest priority rule.
    $rule = reset($applicable_rules);

    // Check if all messages should be blocked for this rule.
    if (!empty($rule['block_all_messages'])) {
      return TRUE;
    }

    // Check if this message type should be blocked.
    if (!empty($rule['blocked_message_types']) && in_array($type, $rule['blocked_message_types'])) {
      return $this->shouldBlockByRouteOrUrl($rule);
    }

    return $this->checkBypassStrategies($message, $type);
  }

  /**
   * Checks if message should be blocked based on route or URL.
   *
   * @param array $rule
   *   The role rule.
   *
   * @return bool
   *   TRUE if should be blocked, FALSE otherwise.
   */
  protected function shouldBlockByRouteOrUrl(array $rule): bool {
    // Check blocked routes.
    $current_route = $this->routeMatch->getRouteName();
    if ($current_route && !empty($rule['blocked_routes']) && in_array($current_route, $rule['blocked_routes'])) {
      return TRUE;
    }

    // Check blocked URLs.
    if (!empty($rule['blocked_urls'])) {
      return $this->urlMatcher->matchesAnyPattern($rule['blocked_urls']);
    }

    return TRUE;
  }

  /**
   * Checks bypass strategies.
   *
   * @param mixed $message
   *   The message.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if should be blocked, FALSE otherwise.
   */
  protected function checkBypassStrategies($message, string $type): bool {
    foreach ($this->bypassStrategyManager->getDefinitions() as $plugin_id => $definition) {
      $strategy = $this->bypassStrategyManager->createInstance($plugin_id);
      if ($strategy->shouldBypass($message, $type, $this->currentUser)) {
        return FALSE;
      }
    }

    return FALSE;
  }

}
