<?php

declare(strict_types=1);

namespace Drupal\message_filter\Service;

/**
 * Interface for unfiltered messenger service.
 */
interface UnfilteredMessengerServiceInterface {

  /**
   * Adds an unfiltered message that should not be filtered.
   *
   * @param mixed $message
   *   The message to add.
   * @param string $type
   *   The message type.
   * @param array $context
   *   Additional context for the message.
   * @param bool $display
   *   Whether to display the message immediately.
   *
   * @return $this
   */
  public function addMessage($message, string $type = 'status', array $context = [], bool $display = TRUE): self;

  /**
   * Adds and displays an unfiltered status message.
   *
   * @param mixed $message
   *   The message to add.
   * @param array $context
   *   Additional context for the message.
   *
   * @return $this
   */
  public function addStatus($message, array $context = []): self;

  /**
   * Adds and displays an unfiltered warning message.
   *
   * @param mixed $message
   *   The message to add.
   * @param array $context
   *   Additional context for the message.
   *
   * @return $this
   */
  public function addWarning($message, array $context = []): self;

  /**
   * Adds and displays an unfiltered error message.
   *
   * @param mixed $message
   *   The message to add.
   * @param array $context
   *   Additional context for the message.
   *
   * @return $this
   */
  public function addError($message, array $context = []): self;

  /**
   * Checks if a message is an unfiltered message.
   *
   * @param mixed $message
   *   The message to check.
   * @param string $type
   *   The message type.
   *
   * @return bool
   *   TRUE if the message is an unfiltered message.
   */
  public function isUnfilteredMessage($message, string $type): bool;

  /**
   * Removes an unfiltered message.
   *
   * @param mixed $message
   *   The message to remove.
   * @param string $type
   *   The message type.
   *
   * @return $this
   */
  public function removeMessage($message, string $type): self;

  /**
   * Gets all unfiltered messages.
   *
   * @return array
   *   Array of unfiltered messages.
   */
  public function getUnfilteredMessages(): array;

  /**
   * Clears all unfiltered messages.
   *
   * @return $this
   */
  public function clearUnfilteredMessages(): self;

}
