<?php

declare(strict_types=1);

namespace Drupal\Tests\message_filter\Unit\Service;

use Drupal\Core\Config\Config;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\message_filter\Service\MessageFilterConfiguration;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for MessageFilterConfiguration service.
 *
 * @group message_filter
 * @coversDefaultClass \Drupal\message_filter\Service\MessageFilterConfiguration
 */
final class MessageFilterConfigurationTest extends UnitTestCase {

  /**
   * The configuration service under test.
   *
   * @var \Drupal\message_filter\Service\MessageFilterConfiguration
   */
  private MessageFilterConfiguration $configuration;

  /**
   * Mock config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  private ConfigFactoryInterface $configFactory;

  /**
   * Mock config object.
   *
   * @var \Drupal\Core\Config\Config|\PHPUnit\Framework\MockObject\MockObject
   */
  private Config $config;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->config = $this->createMock(Config::class);
    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->configFactory
      ->method('get')
      ->with('message_filter.settings')
      ->willReturn($this->config);

    $this->configuration = new MessageFilterConfiguration($this->configFactory);
  }

  /**
   * @covers ::isFilteringEnabled
   */
  public function testIsFilteringEnabledWhenTrue(): void {
    $this->config
      ->method('get')
      ->with('enabled')
      ->willReturn(TRUE);

    $this->assertTrue($this->configuration->isFilteringEnabled());
  }

  /**
   * @covers ::isFilteringEnabled
   */
  public function testIsFilteringEnabledWhenFalse(): void {
    $this->config
      ->method('get')
      ->with('enabled')
      ->willReturn(FALSE);

    $this->assertFalse($this->configuration->isFilteringEnabled());
  }

  /**
   * @covers ::getRoleRulesForUser
   */
  public function testGetRoleRulesForUserWithEnabledRules(): void {
    $account = $this->createMock(AccountInterface::class);
    $account
      ->method('getRoles')
      ->willReturn(['editor', 'content_manager']);

    $role_rules = [
      'editor' => [
        'enabled' => TRUE,
        'priority' => 10,
        'blocked_message_types' => ['status'],
      ],
      'content_manager' => [
        'enabled' => TRUE,
        'priority' => 5,
        'blocked_message_types' => ['warning'],
      ],
      'admin' => [
        'enabled' => TRUE,
        'priority' => 20,
        'blocked_message_types' => ['error'],
      ],
    ];

    $this->config
      ->method('get')
      ->with('role_rules')
      ->willReturn($role_rules);

    $result = $this->configuration->getRoleRulesForUser($account);

    // Should return rules sorted by priority (highest first)
    $this->assertCount(2, $result);
    $this->assertEquals(10, $result[0]['priority']);
    $this->assertEquals(5, $result[1]['priority']);
  }

  /**
   * @covers ::getRoleRulesForUser
   */
  public function testGetRoleRulesForUserWithDisabledRules(): void {
    $account = $this->createMock(AccountInterface::class);
    $account
      ->method('getRoles')
      ->willReturn(['editor']);

    $role_rules = [
      'editor' => [
        'enabled' => FALSE,
        'priority' => 10,
        'blocked_message_types' => ['status'],
      ],
    ];

    $this->config
      ->method('get')
      ->with('role_rules')
      ->willReturn($role_rules);

    $result = $this->configuration->getRoleRulesForUser($account);

    $this->assertEmpty($result);
  }

  /**
   * @covers ::getGlobalConfiguration
   */
  public function testGetGlobalConfiguration(): void {
    $expected_config = [
      'enabled' => TRUE,
      'role_rules' => [],
    ];

    $this->config
      ->method('get')
      ->with()
      ->willReturn($expected_config);

    $result = $this->configuration->getGlobalConfiguration();

    $this->assertEquals($expected_config, $result);
  }

}
