<?php

declare(strict_types=1);

namespace Drupal\Tests\message_filter\Unit\Service;

use Drupal\message_filter\Service\UrlMatcher;
use Drupal\Tests\UnitTestCase;

/**
 * Tests for UrlMatcher service.
 *
 * @group message_filter
 * @coversDefaultClass \Drupal\message_filter\Service\UrlMatcher
 */
final class UrlMatcherTest extends UnitTestCase {

  /**
   * The URL matcher service under test.
   *
   * @var \Drupal\message_filter\Service\UrlMatcher
   */
  private UrlMatcher $urlMatcher;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->urlMatcher = new UrlMatcher();
  }

  /**
   * Data provider for URL matching tests.
   */
  public static function urlMatchingProvider(): array {
    return [
      // Exact matches.
      'exact match' => ['/admin/content', '/admin/content', TRUE],
      'exact match with slash' => ['/admin/', '/admin/', TRUE],
      'exact no match' => ['/admin/content', '/admin/structure', FALSE],

      // Wildcard matches.
      'wildcard at end' => ['/admin/content/add', '/admin/*', TRUE],
      'wildcard at beginning' => ['/some/admin/page', '*/admin/*', TRUE],
      'wildcard in middle' => ['/admin/content/node', '/admin/*/node', TRUE],
      'multiple wildcards' => ['/admin/content/node/add', '/admin/*/node/*', TRUE],
      'wildcard no match' => ['/user/login', '/admin/*', FALSE],

      // Special characters.
      'with query params' => ['/admin/content?page=1', '/admin/*', TRUE],
      'with dots' => ['/admin/content.json', '/admin/*', TRUE],
      'with dashes' => ['/admin/content-types', '/admin/*', TRUE],
    ];
  }

  /**
   * @covers ::matches
   * @dataProvider urlMatchingProvider
   */
  public function testMatches(string $path, string $pattern, bool $expected): void {
    $result = $this->urlMatcher->matches($path, $pattern);
    $this->assertEquals($expected, $result);
  }

  /**
   * @covers ::matchesAny
   */
  public function testMatchesAnyWithMultiplePatterns(): void {
    $path = '/admin/content/add';
    $patterns = [
      '/user/*',
      '/admin/*',
      '/node/*',
    ];

    $result = $this->urlMatcher->matchesAny($path, $patterns);
    $this->assertTrue($result);
  }

  /**
   * @covers ::matchesAny
   */
  public function testMatchesAnyWithNoMatches(): void {
    $path = '/some/other/path';
    $patterns = [
      '/user/*',
      '/admin/*',
      '/node/*',
    ];

    $result = $this->urlMatcher->matchesAny($path, $patterns);
    $this->assertFalse($result);
  }

  /**
   * @covers ::matchesAny
   */
  public function testMatchesAnyWithEmptyPatterns(): void {
    $path = '/admin/content';
    $patterns = [];

    $result = $this->urlMatcher->matchesAny($path, $patterns);
    $this->assertFalse($result);
  }

}
