<?php

namespace Drupal\message_test\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Path\PathValidatorInterface;
use Drupal\Core\Url;
use Drupal\message_test\LoremIpsum;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Allows generation of test messages using the Drupal core messenger service.
 */
class MessageTestForm extends FormBase {

  /**
   * Path validator.
   *
   * @var \Drupal\Core\Path\PathValidatorInterface
   */
  protected $pathValidator;

  /**
   * Constructs a MessageTestForm object.
   *
   * @param \Drupal\Core\Path\PathValidatorInterface $path_validator
   *   The path validator class.
   */
  public function __construct(PathValidatorInterface $path_validator) {
    $this->pathValidator = $path_validator;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('path.validator'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'message_test_form';
  }

  /**
   * {@inheritdoc}
   */
  public function getFormName(): string {
    return 'Generate test messages';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {

    $form['message_table'] = $this->buildTable();

    $form['target_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Target path'),
      '#default_value' => '',
      '#maxlength' => '254',
      '#description' => $this->t('Enter the path to redirect to. Leave blank for front page.'),
    ];

    $form['actions'] = ['#type' => 'actions'];

    $form['actions']['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Generate test messages'),
      '#button_type' => 'primary',
    ];

    return $form;
  }

  /**
   * Build the table with rows for the 3 types of messages.
   */
  public function buildTable() {

    $header = [
      'message_type' => $this->t('Message type'),
      'message_count' => $this->t('Message count'),
      'message_length' => $this->t('Message length (words)'),
    ];

    foreach (['status', 'warning', 'error'] as $messageType) {

      $options[$messageType] = [
        'message_type' => ucfirst($messageType),
        'message_count' => [
          'data' => [
            '#type' => 'number',
            '#name' => $messageType . '_count',
            '#min' => 1,
            '#max' => 10,
            '#value' => 1,
            '#title_display' => 'invisible',
          ],
        ],
        'message_length' => [
          'data' => [
            '#type' => 'number',
            '#name' => $messageType . '_length',
            '#min' => 1,
            '#max' => 200,
            '#value' => 15,
            '#title_display' => 'invisible',
          ],
        ],
      ];
    }

    $table = [
      '#type' => 'tableselect',
      '#caption' => $this->t('Select the message types to generate.'),
      '#header' => $header,
      '#options' => $options,
      '#multiple' => TRUE,
      '#js_select' => TRUE,
    ];

    return $table;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {

    parent::validateForm($form, $form_state);

    // Get the target path given by the user.
    $targetPath = trim($form_state->getValue('target_path'));

    // Target path empty is ok as will default to front page.
    if (empty($targetPath)) {
      return;
    }

    // Entered something so must start with forward slash.
    if (!str_starts_with($targetPath, '/')) {
      $form_state->setErrorByName('metatag_url', $this->t('The target path must begin with /'));
      return FALSE;
    }

    // Get route name from path.
    if (!$this->pathValidator->getUrlIfValid($targetPath)) {
      $form_state->setErrorByName('metatag_url', $this->t('The target path does not exist as an internal Drupal route.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {

    $lipsum = new LoremIpsum();

    $lipsum->words(10);

    // Get the selected message states to send.
    if ($values = $form_state->getValue('message_table')) {
      $selectedTypes = array_filter($values);
    }
    else {
      $selectedTypes = [];
    }

    // Get all input from form.
    $formValues = $form_state->getUserInput();

    // For each state.
    foreach ($selectedTypes as $key => $value) {

      // How many to send? How long?
      $count = $formValues[$key . '_count'];
      $length = $formValues[$key . '_length'];

      for ($i = 0; $i < $count; $i++) {

        switch ($key) {

          case 'status':
            $this->messenger()->addStatus($lipsum->words($length));
            break;

          case 'warning':
            $this->messenger()->addWarning($lipsum->words($length));
            break;

          case 'error':
            $this->messenger()->addError($lipsum->words($length));
            break;
        }
      }
    }

    // Get the redirect path.
    $targetPath = trim($form_state->getValue('target_path'));

    // If empty then redirect to front page.
    if (!$targetPath) {
      $form_state->setRedirect('<front>');
    }

    else {

      $url = Url::fromUri('internal:' . $targetPath);

      $form_state->setRedirectUrl($url);
    }

  }

}
