/**
 * @file
 * Enhanced Status Messages JavaScript functionality.
 * 
 * Provides enhanced animations, auto-hide functionality, and close button handling.
 */

(function ($, Drupal, drupalSettings, once) {

  'use strict';

  /**
   * Enhanced Status Messages behavior.
   */
  Drupal.behaviors.enhancedStatusMessages = {
    attach: function (context, settings) {
      const containers = once('enhanced-status-messages', '.enhanced-status-messages', context);
      
      containers.forEach(function(container) {
        const $container = $(container);
        const config = settings.enhancedStatusMessages || {};
        const autoHideTime = config.autoHideTime || 5000;
        const enableAnimations = config.enableAnimations !== false;
        
        // Initialize messages
        const $messages = $container.find('.enhanced-message-card');
        
        if (enableAnimations) {
          $container.addClass('animations-enabled');
          
          // Show messages with staggered animation
          $messages.each(function(index) {
            const $message = $(this);
            setTimeout(function() {
              $message.addClass('show');
            }, index * 200); // Stagger by 200ms
          });
        } else {
          // Show messages immediately without animation
          $messages.addClass('show').css({
            opacity: 1,
            transform: 'none'
          });
        }
        
        // Auto-hide functionality
        if (autoHideTime > 0) {
          setTimeout(function() {
            hideAllMessages($container, enableAnimations);
          }, autoHideTime);
        }
        
        // Close button functionality
        $messages.find('.enhanced-cross-icon').on('click', function(e) {
          e.preventDefault();
          e.stopPropagation();
          
          const $message = $(this).closest('.enhanced-message-card');
          hideMessage($message, enableAnimations);
        });
        
        // Keyboard accessibility for close buttons
        $messages.find('.enhanced-cross-icon').on('keydown', function(e) {
          if (e.key === 'Enter' || e.key === ' ') {
            e.preventDefault();
            $(this).trigger('click');
          }
        });
      });
    }
  };

  /**
   * Hide a single message.
   * 
   * @param {jQuery} $message - The message element to hide.
   * @param {boolean} enableAnimations - Whether animations are enabled.
   */
  function hideMessage($message, enableAnimations) {
    if (enableAnimations) {
      $message.removeClass('show').addClass('hide');
      setTimeout(function() {
        $message.remove();
      }, 300); // Match CSS transition duration
    } else {
      $message.remove();
    }
  }

  /**
   * Hide all messages in a container.
   * 
   * @param {jQuery} $container - The container with messages.
   * @param {boolean} enableAnimations - Whether animations are enabled.
   */
  function hideAllMessages($container, enableAnimations) {
    const $messages = $container.find('.enhanced-message-card.show');
    
    if (enableAnimations) {
      $messages.each(function(index) {
        const $message = $(this);
        setTimeout(function() {
          hideMessage($message, enableAnimations);
        }, index * 100); // Stagger hide animation
      });
    } else {
      $messages.each(function() {
        hideMessage($(this), enableAnimations);
      });
    }
  }

  /**
   * Show enhanced status messages programmatically.
   * 
   * @param {string} type - Message type (status, warning, error, info).
   * @param {string} message - Message text.
   * @param {string} subText - Optional sub-text.
   * @param {Object} options - Additional options.
   */
  Drupal.enhancedStatusMessages = {
    show: function(type, message, subText, options) {
      options = options || {};
      const config = drupalSettings.enhancedStatusMessages || {};
      
      // Create message HTML
      const messageHtml = Drupal.theme('enhancedStatusMessage', {
        type: type,
        message: message,
        subText: subText || '',
        showWave: config.showWaveBackground !== false
      });
      
      // Find or create container
      let $container = $('.enhanced-status-messages');
      if ($container.length === 0) {
        const position = config.position || 'top-right';
        const animationsClass = config.enableAnimations !== false ? 'animations-enabled' : '';
        const waveClass = config.showWaveBackground === false ? 'hide-wave' : '';
        
        $container = $('<div class="enhanced-status-messages position-' + position + ' ' + animationsClass + ' ' + waveClass + '"></div>');
        $('body').append($container);
      }
      
      // Add message to container
      const $message = $(messageHtml);
      $container.append($message);
      
      // Trigger behavior attachment for new message
      Drupal.behaviors.enhancedStatusMessages.attach($message[0], drupalSettings);
      
      return $message;
    }
  };

})(jQuery, Drupal, drupalSettings, once);
