<?php

/**
 * @file
 * Hooks provided by the Meta Pixel module.
 */

/**
 * @addtogroup hooks
 * @{
 */

/**
 * Alters Meta Pixel event data and user data before tracking.
 *
 * This hook allows modules to modify event parameters and add site-specific
 * user profile fields before data is sent to Meta Pixel and Conversions API.
 *
 * Common use case: Map custom user fields to Meta's user data parameters for
 * improved Event Match Quality.
 *
 * @param array &$event_data
 *   Event data array containing event parameters like content_ids, value,
 *   currency, contents, etc. The event_id is already set for deduplication.
 * @param array &$user_data
 *   User data array with structure:
 *   - profiles: Identity sets keyed by source (account, billing, shipping).
 *   - external_ids: Array of external identifiers.
 *   - emails: Array of email addresses.
 *   - phones: Array of phone numbers.
 * @param array $context
 *   Contextual information including:
 *   - event_name: The event name (e.g., 'Purchase', 'AddToCart').
 *   - user: (UserInterface) Loaded user entity if authenticated.
 *   - order: (OrderInterface) Order entity for commerce events.
 *   - product: (ProductInterface) Product entity for product events.
 *   - product_variation: (ProductVariationInterface) Variation entity.
 *
 * @see \Drupal\meta_pixel\EventCollector::addEvent()
 */
function hook_meta_pixel_event_data_alter(array &$event_data, array &$user_data, array $context = []) {
  // Example: Add custom user profile fields to account profile.
  if (!empty($context['user']) && $context['user'] instanceof \Drupal\user\UserInterface) {
    $user = $context['user'];

    // Initialize account profile structure.
    if (!isset($user_data['profiles']['account'])) {
      $user_data['profiles']['account'] = [];
    }

    $profile = &$user_data['profiles']['account'];

    // Map your custom fields.
    if ($user->hasField('field_first_name') && !$user->get('field_first_name')->isEmpty()) {
      $profile['first_name'] = $user->get('field_first_name')->value;
    }

    if ($user->hasField('field_last_name') && !$user->get('field_last_name')->isEmpty()) {
      $profile['last_name'] = $user->get('field_last_name')->value;
    }

    // Address field (using Address field type).
    if ($user->hasField('field_address') && !$user->get('field_address')->isEmpty()) {
      /** @var \Drupal\address\Plugin\Field\FieldType\AddressItem $address */
      $address = $user->get('field_address')->first();
      if ($address) {
        $profile['city'] = $address->getLocality();
        $profile['state'] = $address->getAdministrativeArea();
        $profile['zip_code'] = $address->getPostalCode();
        $profile['country_code'] = $address->getCountryCode();
      }
    }

    // Phone numbers go in phones array (not profile).
    if ($user->hasField('field_phone') && !$user->get('field_phone')->isEmpty()) {
      $user_data['phones'][] = $user->get('field_phone')->value;
    }

  }

  // Example: Add custom event data.
  if ($context['event_name'] === 'Purchase' && !empty($context['order'])) {
    // Add custom business logic to event data.
    $event_data['custom_field'] = 'custom_value';
  }
}

/**
 * @} End of "addtogroup hooks".
 */
