<?php

declare(strict_types=1);

namespace Drupal\meta_pixel_commerce\EventSubscriber;

use Drupal\commerce_cart\Event\CartEvents;
use Drupal\commerce_cart\Event\CartOrderItemAddEvent;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\meta_pixel\EventCollector;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Fires Meta Pixel events on cart actions.
 */
final class CartSubscriber implements EventSubscriberInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  private EventCollector $eventCollector;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private AccountProxyInterface $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private RequestStack $requestStack;

  /**
   * Constructs a CartSubscriber object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(
    EventCollector $eventCollector,
    AccountProxyInterface $currentUser,
    RequestStack $requestStack,
  ) {
    $this->eventCollector = $eventCollector;
    $this->currentUser = $currentUser;
    $this->requestStack = $requestStack;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      CartEvents::CART_ORDER_ITEM_ADD => 'onAdd',
    ];
  }

  /**
   * Fires AddToCart event when item is added to cart.
   *
   * @param \Drupal\commerce_cart\Event\CartOrderItemAddEvent $cart_event
   *   The cart event.
   */
  public function onAdd(CartOrderItemAddEvent $cart_event): void {
    $order_item = $cart_event->getOrderItem();
    $purchased_entity = $order_item->getPurchasedEntity();

    if (!$purchased_entity) {
      return;
    }

    $order = $cart_event->getCart();

    // If the acting user is not the customer, an admin is placing the order
    // so we don't want to track.
    if ($order->getCustomerId() != $this->currentUser->id()) {
      return;
    }

    // Detect if this is an AJAX request.
    $request = $this->requestStack->getCurrentRequest();
    $is_ajax = $request && $request->isXmlHttpRequest();

    // Add context for caching and alter hook.
    $data['item'] = $order_item;
    $data['purchased_entity'] = $purchased_entity;
    $data['order'] = $order;

    // Use session storage only for non-AJAX.
    $use_session = !$is_ajax;

    // Fire the event.
    $this->eventCollector->addEvent('commerce_add_to_cart', $data, $use_session);
  }

}
