<?php

declare(strict_types=1);

namespace Drupal\meta_pixel_commerce\Plugin\MetaPixelEvent;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\meta_pixel\Attribute\MetaPixelEvent;

/**
 * Builds AddToCart event data when items are added to the cart.
 *
 * @see \Drupal\meta_pixel_commerce\EventSubscriber\CartSubscriber
 */
#[MetaPixelEvent(
  id: 'commerce_add_to_cart',
  event_name: 'AddToCart',
  label: new TranslatableMarkup('Add to Cart (Commerce)'),
  description: new TranslatableMarkup('Fires when a product is added to the cart.'),
  weight: 0,
)]
class AddToCartEvent extends CommerceMetaPixelEventPluginBase {

  /**
   * {@inheritdoc}
   */
  public function applies(array $data): bool {
    // Requires 'item' key with an order item entity.
    return !empty($data['item']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildEventData(): array {
    $event_data = parent::buildEventData();
    $data = $this->getData();
    $order_item = $data['item'] ?? NULL;
    if (!$order_item) {
      return $event_data;
    }
    $order = $order_item->getOrder();
    $purchased_entity = $order_item->getPurchasedEntity();
    if (!$purchased_entity) {
      return $event_data;
    }

    // Get SKU.
    $sku = $this->getSku($purchased_entity);

    // Get price from order item (includes quantity).
    $total_price = $order_item->getTotalPrice();
    $unit_price = $order_item->getUnitPrice();

    if (!$total_price || !$unit_price) {
      return $event_data;
    }

    // Build Meta-formatted event data.
    $item_data = [
      'content_ids' => [$sku],
      'content_type' => 'product',
      'value' => (float) $total_price->getNumber(),
      'currency' => $total_price->getCurrencyCode(),
      'order_id' => !empty($order) ? $order->id() : NULL,
      'contents' => [
        [
          'id' => $sku,
          'quantity' => (int) $order_item->getQuantity(),
          'item_price' => (float) $unit_price->getNumber(),
          'title' => $order_item->getTitle(),
        ],
      ],
    ];
    return array_merge($event_data, $item_data);
  }

}
