<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\meta_pixel\EventCollector;
use Drupal\meta_pixel\MetaCapiClient;
use Drupal\meta_pixel\MetaPixelEvent;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;

/**
 * Controller for CAPI PageView event tracking.
 *
 * Handles server-side PageView tracking via AJAX callback from browser.
 * The browser pixel fires first with an event_id, then this controller
 * sends the same event to CAPI with the matching event_id for deduplication.
 */
class PageViewController extends ControllerBase {

  /**
   * The event collector service.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  protected EventCollector $eventCollector;

  /**
   * The Meta CAPI client service.
   *
   * @var \Drupal\meta_pixel\MetaCapiClient
   */
  protected MetaCapiClient $metaCapiClient;

  /**
   * Constructs a PageViewController object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector service.
   * @param \Drupal\meta_pixel\MetaCapiClient $metaCapiClient
   *   The Meta CAPI client service.
   */
  public function __construct(EventCollector $eventCollector, MetaCapiClient $metaCapiClient) {
    $this->eventCollector = $eventCollector;
    $this->metaCapiClient = $metaCapiClient;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('meta_pixel.event_collector'),
      $container->get('meta_pixel.meta_capi_client'),
    );
  }

  /**
   * Sends PageView event to Meta Conversions API.
   *
   * This controller handles server-side PageView tracking via AJAX:
   * 1. Browser pixel fires PageView with event_id.
   * 2. Browser makes AJAX request to this endpoint with same event_id.
   * 3. Server sends CAPI event using matching event_id for deduplication.
   *
   * We bypass addEvent() here because the event_id is predetermined by
   * the browser, and we only want CAPI tracking (browser already fired).
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request object containing event_id and URL parameters.
   *
   * @return \Symfony\Component\HttpFoundation\JsonResponse
   *   JSON response indicating success or failure.
   */
  public function pageView(Request $request): JsonResponse {
    if ($this->eventCollector->isCapiEnabled('page_view') && _meta_pixel_visibility_roles($this->currentUser())) {
      try {
        // Send the event to Meta.
        $event_data = [
          'event_name' => 'PageView',
          'event_id' => $request->query->get('event_id') ?? NULL,
          'event_source_url' => $request->query->get('url') ?? NULL,
        ];

        if (empty($event_data['event_id']) || empty($event_data['event_source_url'])) {
          return new JsonResponse([
            'success' => FALSE,
            'reason' => 'missing_event_data',
          ]);
        }

        $user_data = $this->eventCollector->buildUserData($event_data);

        // Allow other modules to alter the event before sending to CAPI.
        $context['event_name'] = $event_data['event_name'];
        $this->moduleHandler()->alter('meta_pixel_event_data', $event_data, $user_data, $context);

        $pixel_event = new MetaPixelEvent('PageView', $event_data, $user_data);
        $capi_event = $this->eventCollector->prepareCapiEvent($pixel_event);
        $this->metaCapiClient->sendRequest([$capi_event]);

        return new JsonResponse(['success' => TRUE]);
      }
      catch (\Exception $e) {
        return new JsonResponse([
          'success' => FALSE,
          'reason' => 'exception',
          'error' => $e->getMessage(),
        ]);
      }
    }
    return new JsonResponse([
      'success' => FALSE,
    ]);
  }

}
