<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use FacebookAds\Object\ServerSide\Event;

/**
 * Interface for the Meta Pixel event collector service.
 *
 * The event collector is the central coordinator for Meta tracking events,
 * managing both browser-side pixel and server-side Conversions API tracking
 * with proper event deduplication.
 */
interface EventCollectorInterface {

  /**
   * Adds an event to be tracked by Meta Pixel.
   *
   * This is the main entry point for all Meta tracking events, coordinating
   * dual-channel tracking across browser-side pixel (fbq) and server-side
   * Conversions API.
   *
   * @param string $plugin_id
   *   The Meta event plugin ID (e.g., 'commerce_view_content',
   *   'commerce_purchase').
   * @param array $data
   *   Context data needed to build the event (entities, etc.).
   * @param bool $use_session
   *   Whether to use session storage for browser events (for POST-redirect-GET
   *   patterns). Default FALSE uses static storage for immediate page render.
   */
  public function addEvent(string $plugin_id, array $data = [], bool $use_session = FALSE): MetaPixelEvent|null;

  /**
   * Gets queued browser pixel events.
   *
   * Returns events ready for transmission to the browser via JavaScript.
   * Safe to call during hook_page_attachments or ResponseSubscriber
   * (before response sent). May access session storage.
   *
   * @return \Drupal\meta_pixel\MetaPixelEvent[]
   *   Array of browser events, deduplicated by event_id.
   */
  public function getBrowserEvents(): array;

  /**
   * Gets queued CAPI events.
   *
   * Returns events ready for server-side transmission to Meta's API.
   * Safe to call during kernel.terminate because it only uses static
   * storage, never session.
   *
   * @return \Drupal\meta_pixel\MetaPixelEvent[]
   *   Array of CAPI events, deduplicated by event_id.
   */
  public function getCapiEvents(): array;

  /**
   * Prepares a MetaPixelEvent for transmission to CAPI.
   *
   * Converts a MetaPixelEvent object into a Facebook SDK Event object
   * ready for server-side transmission.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The Meta Pixel event to prepare.
   *
   * @return \FacebookAds\Object\ServerSide\Event|null
   *   The prepared Facebook SDK Event object, or NULL if CAPI is not enabled.
   */
  public function prepareCapiEvent(MetaPixelEvent $event): Event|null;

  /**
   * Builds user identification data for Meta tracking events.
   *
   * Collects user identity data from available sources to maximize Event
   * Match Quality. Can be called by alter hook implementations to access
   * or extend user data collection.
   *
   * @param array $event_data
   *   Event data that may contain an order object.
   * @param array $context
   *   Context array with entities. Will be updated with loaded user/order
   *   entities.
   *
   * @return array
   *   User data with structure:
   *   - profiles: Array keyed by source (account, billing, shipping).
   *   - external_ids: Array of external identifiers.
   *   - emails: Array of email addresses.
   *   - phones: Array of phone numbers.
   */
  public function buildUserData(array $event_data = [], array &$context = []): array;

  /**
   * Checks if CAPI tracking is enabled for a specific event plugin.
   *
   * @param string $plugin_id
   *   The event plugin ID (e.g., 'commerce_view_content').
   *
   * @return bool
   *   TRUE if CAPI tracking is enabled for this event, FALSE otherwise.
   */
  public function isCapiEnabled(string $plugin_id): bool;

  /**
   * Checks if browser pixel tracking is enabled for a specific event plugin.
   *
   * @param string $plugin_id
   *   The event plugin ID (e.g., 'commerce_view_content').
   *
   * @return bool
   *   TRUE if browser tracking is enabled for this event, FALSE otherwise.
   */
  public function isBrowserEnabled(string $plugin_id): bool;

  /**
   * Checks if any tracking method is enabled for a specific event plugin.
   *
   * Convenience method to check if either browser pixel or CAPI tracking
   * is enabled.
   *
   * @param string $plugin_id
   *   The event plugin ID (e.g., 'commerce_view_content').
   *
   * @return bool
   *   TRUE if either browser or CAPI tracking is enabled, FALSE if both
   *   are disabled.
   */
  public function isAnyEnabled(string $plugin_id): bool;

}
