<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\EventSubscriber;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\meta_pixel\Ajax\MetaPixelTrackCommand;
use Drupal\meta_pixel\EventCollector;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\ResponseEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Adds Meta Pixel events to AJAX responses.
 */
final class ResponseSubscriber implements EventSubscriberInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  private EventCollector $eventCollector;

  /**
   * Constructs a ResponseSubscriber object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector.
   */
  public function __construct(EventCollector $eventCollector) {
    $this->eventCollector = $eventCollector;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::RESPONSE => ['onResponse', 10],
    ];
  }

  /**
   * Fires Meta Pixel events for AJAX responses.
   *
   * @param \Symfony\Component\HttpKernel\Event\ResponseEvent $event
   *   The response event.
   */
  public function onResponse(ResponseEvent $event): void {
    // dsm($event->isMainRequest(), '$event->isMainRequest()');
    // Only act on main request.
    if (!$event->isMainRequest()) {
      return;
    }

    $response = $event->getResponse();

    // Only handle AJAX responses.
    if (!$response instanceof AjaxResponse) {
      return;
    }

    // dsm($response);
    // Get all queued events.
    foreach ($this->eventCollector->getBrowserEvents() as $pixel_event) {
      // Get event name and data from MetaPixelEvent object.
      $event_name = $pixel_event->getName();
      $event_data = $pixel_event->getEventData();
      $event_id = $pixel_event->getEventId();

      // Pass event data to the ajax MetaPixelTrackCommand. The command
      // handles extracting event_id and cleaning data for browser.
      $response->addCommand(new MetaPixelTrackCommand($event_name, $event_data, $event_id));
    }
  }

}
