<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\Form;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use FacebookAds\Object\ServerSide\ActionSource;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configuration form for Meta Pixel settings.
 */
class SettingsForm extends ConfigFormBase {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    $instance = parent::create($container);
    $instance->moduleHandler = $container->get('module_handler');
    $instance->entityTypeManager = $container->get('entity_type.manager');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['meta_pixel.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'meta_pixel_settings_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('meta_pixel.settings');

    $form['info'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure Meta Pixel tracking for your site. The pixel enables both browser-side tracking and server-side Conversions API.') . '</p>',
    ];

    $form['pixel_id'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Meta Pixel ID'),
      '#default_value' => $config->get('pixel_id'),
      '#description' => $this->t('Your Meta Pixel ID from Meta Events Manager. Example: 1234567890123456'),
      '#required' => FALSE,
      '#size' => 30,
      '#config_target' => 'meta_pixel.settings:pixel_id',
    ];

    $form['test_event_code'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Test event code'),
      '#description' => $this->t('This is available at <a href="@meta" target="_blank">@meta</a> in the form "TEST1234"', [
        '@meta' => 'https://business.facebook.com/events_manager2',
      ]),
      '#config_target' => 'meta_pixel.settings:test_event_code',
      '#size' => 30,
    ];

    // Create vertical tabs container.
    $form['settings'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Settings'),
      '#title_display' => 'invisible',
    ];

    // CAPI settings.
    $form['capi'] = [
      '#type' => 'details',
      '#title' => $this->t('Meta Conversions API'),
      '#group' => 'settings',
      '#open' => FALSE,
    ];

    $form['capi']['access_token'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Conversions API Access token'),
      '#description' => $this->t('You can generate this at <a href="@meta" target="_blank">@meta</a>
        <ul>
          <li>Select the appropriate App under <em>Facebook App</em></li>
          <li>Then from <em>User or Page</em> select your listed page from <em>Page Access Tokens</em></li>
        </ul>
        ', ['@meta' => 'https://developers.facebook.com/tools/explorer/']),
      '#config_target' => 'meta_pixel.settings:capi.access_token',
      '#size' => 60,
      '#maxlength' => 255,
    ];

    $actionSources = ActionSource::getInstance()->getValues();
    $form['capi']['default_action_source'] = [
      '#type' => 'select',
      '#title' => $this->t('Default event action source'),
      '#description' => $this->t('The default action source when the event does not specify one.'),
      '#options' => array_combine($actionSources, $actionSources),
      '#config_target' => 'meta_pixel.settings:capi.default_action_source',
      '#required' => TRUE,
    ];

    $form['capi']['filter_expired_fbc'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Filter expired fbc values'),
      '#description' => $this->t('Prevent sending fbc (Facebook Click ID) values older than 90 days to Conversions API. Meta may flag expired fbc values in diagnostics. Enable this setting if you see "expired fbclid" warnings in Events Manager. <strong>Experimental:</strong> This filters server-side and may affect attribution for returning customers.'),
      '#config_target' => 'meta_pixel.settings:capi.filter_expired_fbc',
    ];

    $form['capi']['enable_logging'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable logging'),
      '#description' => $this->t("Log detailed information about Conversions API requests and responses to Drupal's watchdog system. Useful for debugging event data and verifying API communication. Disable in production."),
      '#config_target' => 'meta_pixel.settings:capi.enable_logging',
    ];

    $form['param_builder'] = [
      '#type' => 'details',
      '#title' => $this->t('Parameter Builder'),
      '#group' => 'settings',
      '#open' => FALSE,
    ];

    $form['param_builder']['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t("Meta's Parameter Builder library improves cookie handling and provides IPv6 support for better Event Match Quality.") . '</p>',
    ];

    $form['param_builder']['param_builder_enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Parameter Builder'),
      '#description' => $this->t('Requires <code>facebook/capi-param-builder-php</code> composer package.'),
      '#default_value' => $config->get('param_builder.enabled') ?? FALSE,
      '#config_target' => 'meta_pixel.settings:param_builder.enabled',
    ];

    $form['param_builder']['param_builder_domains'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Domains'),
      '#description' => $this->t('Comma-separated list. Example: <code>example.com, example.co.uk</code>'),
      '#default_value' => $config->get('param_builder.domains') ?? '',
      '#config_target' => 'meta_pixel.settings:param_builder.domains',
      '#placeholder' => 'example.com',
      '#states' => [
        'visible' => [
          ':input[name="param_builder_enabled"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Privacy settings.
    $form['privacy'] = [
      '#type' => 'details',
      '#title' => $this->t('Privacy Settings'),
      '#group' => 'settings',
      '#open' => FALSE,
    ];

    $form['privacy']['disable_advanced_matching'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable advanced matching'),
      '#default_value' => $config->get('privacy.disable_advanced_matching') ?? FALSE,
      '#description' => $this->t('If enabled, for enhanced privacy if Facebook Pixel user opt-out code "<i>window[\'fb-disable\']</i>" is true, the Facebook pixel module will not execute the Facebook Pixel tracking code on your site. Furthermore provides the global JavaScript function "fbOptout()" to set an opt-out cookie if called.'),
      '#config_target' => 'meta_pixel.settings:privacy.disable_advanced_matching',
    ];

    $form['privacy']['respect_dnt'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Universal web tracking opt-out'),
      '#default_value' => $config->get('privacy.respect_dnt') ?? FALSE,
      '#description' => $this->t('If enabled, if a user has <a href="@donottrack">Do-Not-Track</a> enabled in the browser, the Facebook Pixel module will not execute the tracking code on your site. Compliance with Do Not Track could be purely voluntary, enforced by industry self-regulation, or mandated by state or federal law. Please accept your visitors privacy. If they have opt-out from tracking and advertising, you should accept their personal decision.', ['@donottrack' => 'https://www.eff.org/issues/do-not-track']),
      '#config_target' => 'meta_pixel.settings:privacy.respect_dnt',
    ];

    $form['privacy']['eu_cookie_compliance'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('EU Cookie Compliance integration'),
      '#description' => $this->t('If enabled, the Facebook Pixel module will not track users as long as we do not have their consent. This option is designed to work with the module <a href="@eu_cookie_compliance">Eu Cookie Compliance</a>. Technically it checks for Drupal.eu_cookie_compliance.hasAgreed(). <strong>Important:</strong> Set "Script scope" to "Header" in the EU Cookie Compliance settings for this to work.', ['@eu_cookie_compliance' => 'https://www.drupal.org/project/eu_cookie_compliance']),
      '#default_value' => $this->moduleHandler->moduleExists('eu_cookie_compliance') ? $config->get('privacy.eu_cookie_compliance') : 0,
      '#disabled' => !$this->moduleHandler->moduleExists('eu_cookie_compliance'),
      '#config_target' => 'meta_pixel.settings:privacy.eu_cookie_compliance',
    ];

    $form['privacy']['disable_noscript'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable noscript fallback image, which does not respect any of these privacy features.'),
      '#default_value' => $config->get('privacy.disable_noscript') ?? FALSE,
      '#description' => $this->t('Disable the noscript fallback tracking image. Recommended for privacy compliance.'),
      '#config_target' => 'meta_pixel.settings:privacy.disable_noscript',
    ];

    $form['privacy']['exclude_params'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Exclude params'),
      '#description' => $this->t("A comma-separated list of parameters to strip from URLs before sending to Meta. Meta may flag your account if they detect personally identifiable information being transmitted. For example, latitude and longitude parameters in URLs on store locator search pages have previously triggered account warnings. This setting allows you to exclude specific parameters from the event_source_url when sending events to Meta."),
      '#config_target' => 'meta_pixel.settings:privacy.exclude_params',
      '#default_value' => $config->get('privacy.exclude_params') ?? '',
    ];

    // Visibility settings - User Roles.
    $form['visibility_roles'] = [
      '#type' => 'details',
      '#title' => $this->t('User Roles'),
      '#group' => 'settings',
      '#open' => FALSE,
    ];

    $form['visibility_roles']['user_role_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Track users by role'),
      '#options' => [
        'all_roles' => $this->t('Track all users except selected roles'),
        'selected_roles' => $this->t('Track only selected roles'),
      ],
      '#default_value' => $config->get('visibility.user_role_mode') ?? 'all_roles',
      '#config_target' => 'meta_pixel.settings:visibility.user_role_mode',
    ];

    $role_options = array_map(function ($role) {
      return $role->label();
    }, $this->entityTypeManager->getStorage('user_role')->loadMultiple());

    $form['visibility_roles']['user_role_roles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Roles'),
      '#options' => $role_options,
      '#default_value' => $config->get('visibility.user_role_roles') ?? [],
      '#description' => $this->t('Select which roles should be affected by the setting above.'),
      '#config_target' => 'meta_pixel.settings:visibility.user_role_roles',
    ];

    // Visibility settings - Pages.
    $form['visibility_pages'] = [
      '#type' => 'details',
      '#title' => $this->t('Page Visibility'),
      '#open' => FALSE,
      '#description' => $this->t('Page visibility settings control only where the browser-side JavaScript pixel is loaded. Server-side Conversions API events are not affected by these settings and will fire on all pages where events are triggered.'),
      '#group' => 'settings',
    ];

    $form['visibility_pages']['request_path_mode'] = [
      '#type' => 'radios',
      '#title' => $this->t('Track pages'),
      '#options' => [
        'all_pages' => $this->t('Track all pages except listed pages'),
        'listed_pages' => $this->t('Track only listed pages'),
      ],
      '#default_value' => $config->get('visibility.request_path_mode') ?? 'all_pages',
      '#config_target' => 'meta_pixel.settings:visibility.request_path_mode',
    ];

    $form['visibility_pages']['request_path_pages'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Pages'),
      '#default_value' => $config->get('visibility.request_path_pages') ?? '',
      '#description' => $this->t("Specify pages by using their paths. Enter one path per line. The '*' character is a wildcard. An example path is %user-wildcard for every user page. %front is the front page.", [
        '%user-wildcard' => '/user/*',
        '%front' => '<front>',
      ]),
      '#config_target' => 'meta_pixel.settings:visibility.request_path_pages',
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state) {
    parent::validateForm($form, $form_state);

    $enabled = $form_state->getValue('param_builder_enabled');
    $domains = $form_state->getValue('param_builder_domains');

    if ($enabled && empty(trim((string) $domains))) {
      $form_state->setErrorByName('param_builder_domains',
        $this->t('Domains are required when Parameter Builder is enabled.')
      );
    }
  }

}
