<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Component\Utility\UrlHelper;
use Drupal\Core\Cache\CacheBackendInterface;
use Drupal\Core\Config\ConfigFactory;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\meta_pixel\Logger\FacebookLogger;
use FacebookAds\Api;
use FacebookAds\Object\ServerSide\Content;
use FacebookAds\Object\ServerSide\CustomData;
use FacebookAds\Object\ServerSide\Event;
use FacebookAds\Object\ServerSide\EventRequest;
use FacebookAds\Object\ServerSide\UserData;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Meta Conversions API client service.
 *
 * Handles server-side event transmission to Meta's Conversions API,
 * including event building, user data enrichment, and batch requests.
 */
class MetaCapiClient {

  /**
   * Facebook API instance.
   *
   * @var \FacebookAds\Api
   */
  protected Api $api;

  /**
   * The Meta Pixel settings configuration.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected ImmutableConfig $config;

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The cache backend service.
   *
   * @var \Drupal\Core\Cache\CacheBackendInterface
   */
  protected CacheBackendInterface $cache;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The logger service.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The Facebook SDK logger adapter.
   *
   * @var \Drupal\meta_pixel\Logger\FacebookLogger
   */
  protected FacebookLogger $facebookLogger;

  /**
   * The Parameter Builder factory service.
   *
   * @var \Drupal\meta_pixel\MetaParamBuilderFactory
   */
  protected MetaParamBuilderFactory $paramBuilderFactory;

  /**
   * Constructs a MetaCapiClient object.
   *
   * @param \Drupal\Core\Config\ConfigFactory $config_factory
   *   The configuration factory.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Drupal\Core\Cache\CacheBackendInterface $cache
   *   The cache backend service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack service.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger service.
   * @param \Drupal\meta_pixel\Logger\FacebookLogger $facebook_logger
   *   The Facebook SDK logger adapter.
   * @param \Drupal\meta_pixel\MetaParamBuilderFactory $param_builder_factory
   *   The Parameter Builder factory service.
   */
  public function __construct(
    ConfigFactory $config_factory,
    ModuleHandlerInterface $module_handler,
    CacheBackendInterface $cache,
    RequestStack $request_stack,
    LoggerInterface $logger,
    FacebookLogger $facebook_logger,
    MetaParamBuilderFactory $param_builder_factory,
  ) {
    $this->config = $config_factory->get('meta_pixel.settings');
    $this->moduleHandler = $module_handler;
    $this->cache = $cache;
    $this->requestStack = $request_stack;
    $this->logger = $logger;
    $this->facebookLogger = $facebook_logger;
    $this->paramBuilderFactory = $param_builder_factory;

    $accessToken = $this->config->get('capi.access_token');
    if ($accessToken) {
      $this->api = Api::init(
        NULL,
        NULL,
        $accessToken,
        FALSE
      );

      if ($this->config->get('capi.enable_logging')) {
        $this->api->setLogger($this->facebookLogger);
      }
    }
  }

  /**
   * Checks if the Meta Conversions API is properly configured and enabled.
   *
   * @return bool
   *   TRUE if API is configured with access token and pixel ID, FALSE
   *   otherwise.
   */
  public function isEnabled(): bool {
    // Can't send requests unless API is configured.
    if (!isset($this->api)) {
      return FALSE;
    }

    // Need a pixel ID.
    $pixel_id = $this->config->get('pixel_id');
    if (empty($pixel_id)) {
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Builds a Facebook SDK Event object from provided data.
   *
   * Converts Drupal-formatted event data into a Facebook SDK Event object,
   * enriching it with user data, cookie tracking, client information, and
   * custom event parameters. Automatically adds missing required fields like
   * event_time, action_source, and event_source_url.
   *
   * @param array $eventData
   *   Event metadata containing:
   *   - event_name: (required) The Meta event name (e.g., 'Purchase').
   *   - event_id: (required) Unique event identifier for deduplication.
   *   - event_time: (optional) Unix timestamp, defaults to current time.
   *   - action_source: (optional) Defaults to configured value.
   *   - event_source_url: (optional) Defaults to current request URL.
   * @param array $userData
   *   User identification data array with keys matching Meta's UserData
   *   format (email, phone, external_id, etc.). Will be enriched with
   *   client IP, user agent, and Facebook cookies (_fbc, _fbp).
   * @param array $customData
   *   Event-specific custom data (value, currency, contents, etc.).
   *
   * @return \FacebookAds\Object\ServerSide\Event|null
   *   The prepared Facebook SDK Event object, or NULL if API is not enabled
   *   or event_name is missing.
   */
  public function buildEvent(array $eventData, array $userData = [], array $customData = []): Event|null {
    if (!$this->isEnabled()) {
      return NULL;
    }

    if (empty($eventData['event_name'])) {
      $this->logger->error('Missing event_name in Meta request, called from @trace', [
        '@trace' => debug_backtrace(!DEBUG_BACKTRACE_PROVIDE_OBJECT | DEBUG_BACKTRACE_IGNORE_ARGS, 2)[1]['function'],
      ]);
      return NULL;
    }

    $currentRequest = $this->requestStack->getCurrentRequest();
    if (!$currentRequest) {
      return NULL;
    }

    // Get tracking data (fbc, fbp, IP) via Parameter Builder or fallback.
    $tracking = $this->getTrackingData();

    // Build UserData object.
    $userDataObject = new UserData($userData);

    // Set fbc if available.
    if (!empty($tracking['fbc'])) {
      $userDataObject->setFbc($tracking['fbc']);
    }

    // Set fbp if available.
    if (!empty($tracking['fbp'])) {
      $userDataObject->setFbp($tracking['fbp']);
    }

    // Set client IP address.
    if (!empty($tracking['ip_address'])) {
      $userDataObject->setClientIpAddress($tracking['ip_address']);
    }

    // Set client user agent.
    if (!isset($userData['client_user_agent'])) {
      $userDataObject->setClientUserAgent($currentRequest->headers->get('User-Agent'));
    }

    // Handle structural differences between browser pixel and CAPI.
    //
    // The Facebook SDK for CAPI requires custom event properties to be
    // separated from standard Meta parameters. Standard parameters (value,
    // currency, content_ids, etc.) go in the main array, while any custom
    // properties must be nested within a 'custom_properties' array.
    //
    // In contrast, the browser pixel (fbq) accepts all properties in a flat
    // array. This function extracts non-standard parameters and restructures
    // them for CAPI.
    $customDataObject = new CustomData($customData);

    // Filter out valid standard Meta parameters from the custom data array.
    $valid_custom_data = $customDataObject->attributeMap();
    $custom_properties = array_diff_key($customData, $valid_custom_data);

    // Get any custom properties that may have already been set directly.
    $existing_custom = $customDataObject->getCustomProperties() ?? [];

    // Merge previously set custom properties with newly extracted ones.
    $merged_custom = array_merge($existing_custom, $custom_properties);
    if ($merged_custom) {
      $customDataObject->setCustomProperties($merged_custom);
    }

    if (isset($customData['content'])) {
      $content = new Content($customData['content']);
      $customDataObject->setContents([$content]);
    }

    // Get event source URL (provided or current request) and strip excluded
    // params.
    $source_url = !empty($eventData['event_source_url'])
      ? $eventData['event_source_url']
      : $currentRequest->getSchemeAndHttpHost() . $currentRequest->getRequestUri();
    $eventData['event_source_url'] = $this->stripExcludedParams($source_url);

    // Combine all the data into an event.
    $event = new Event($eventData);
    if (!isset($eventData['event_time'])) {
      $event->setEventTime(time());
    }
    if (!isset($eventData['action_source'])) {
      $event->setActionSource($this->config->get('capi.default_action_source'));
    }
    $event->setUserData($userDataObject)
      ->setCustomData($customDataObject);

    return $event;
  }

  /**
   * Gets tracking data (fbc, fbp, IP) from Parameter Builder or fallback.
   *
   * Attempts to use Parameter Builder when enabled for improved cookie handling
   * and IPv6 support. Falls back to direct cookie reading if Parameter Builder
   * is disabled or fails.
   *
   * Parameter Builder provides three tracking values:
   * - fbc: Facebook Click ID from ad clicks
   * - fbp: Facebook Browser ID for user identification
   * - client_ip_address: Client IP address (IPv6 preferred, IPv4 fallback)
   *
   * Note: Parameter Builder also provides getNormalizedAndHashedPII() for PII
   * normalization, but this is redundant since Facebook's Business SDK UserData
   * class automatically handles normalization and hashing.
   *
   * @return array
   *   Associative array with keys:
   *   - fbc: Facebook Click ID (string|null)
   *   - fbp: Facebook Browser ID (string|null)
   *   - ip_address: Client IP address, IPv6 preferred (string|null)
   */
  protected function getTrackingData(): array {
    $currentRequest = $this->requestStack->getCurrentRequest();
    if (!$currentRequest) {
      return [
        'fbc' => NULL,
        'fbp' => NULL,
        'ip_address' => NULL,
      ];
    }

    $fbc = NULL;
    $fbp = NULL;
    $ipAddress = NULL;

    // Check if Parameter Builder is enabled.
    $use_param_builder = $this->config->get('param_builder.enabled');

    if ($use_param_builder) {
      try {
        // Process the request to extract fbc, fbp, and IP.
        // This analyzes:
        // - URL query params (fbclid for fbc generation)
        // - Cookies (_fbc, _fbp)
        // - Request headers (X-Forwarded-For, referer)
        // - Remote address (for IPv6 detection)
        $this->paramBuilderFactory->processRequest();

        // Get the three tracking values Parameter Builder provides.
        $fbc = $this->paramBuilderFactory->getFbc();
        $fbp = $this->paramBuilderFactory->getFbp();
        $ipAddress = $this->paramBuilderFactory->getClientIpAddress();

        // Log Parameter Builder results if debugging enabled.
        if ($this->config->get('capi.enable_logging')) {
          $ip_type = (!empty($ipAddress) && strpos($ipAddress, ':') !== FALSE) ? 'IPv6' : 'IPv4';
          $this->logger->info('Parameter Builder - fbc: @fbc | fbp: @fbp | IP: @ip (@type)', [
            '@fbc' => $fbc ?? 'NULL',
            '@fbp' => $fbp ?? 'NULL',
            '@ip' => $ipAddress ?? 'NULL',
            '@type' => $ip_type,
          ]);
        }
      }
      catch (\Exception $e) {
        $this->logger->error('Parameter Builder failed, falling back to cookie reading: @message', [
          '@message' => $e->getMessage(),
        ]);

        // Fallback to direct cookie reading on error.
        $fbc = $currentRequest->cookies->get('_fbc');
        $fbp = $currentRequest->cookies->get('_fbp');
        $ipAddress = $currentRequest->getClientIp();
      }
    }
    else {
      // Direct cookie reading (existing implementation).
      $fbc = $currentRequest->cookies->get('_fbc');
      $fbp = $currentRequest->cookies->get('_fbp');
      $ipAddress = $currentRequest->getClientIp();

      // Apply legacy fbc validation if enabled.
      if (!empty($fbc) && $this->config->get('capi.filter_expired_fbc')) {
        if (!$this->isValidFbc($fbc)) {
          $fbc = NULL;
        }
      }
    }

    return [
      'fbc' => $fbc,
      'fbp' => $fbp,
      'ip_address' => $ipAddress,
    ];
  }

  /**
   * Sends events to Meta Conversions API.
   *
   * Transmits one or more prepared Event objects to Meta's server-side
   * tracking API. Events should be pre-built using buildEvent() before
   * calling this method.
   *
   * @param \FacebookAds\Object\ServerSide\Event[] $events
   *   Array of Facebook SDK Event objects to send.
   * @param string|null $testEventCode
   *   Optional test event code for debugging in Meta Events Manager.
   *   If not provided, uses the configured test event code from settings.
   *
   * @throws \Exception
   *   When the API request fails, logged but not re-thrown.
   */
  public function sendRequest(array $events, ?string $testEventCode = NULL): void {
    if (!$this->isEnabled()) {
      return;
    }

    // Use configured test event code if not provided.
    if (empty($testEventCode)) {
      $testEventCode = $this->config->get('test_event_code');
    }

    $request = (new EventRequest($this->config->get('pixel_id')))
      ->setEvents($events)
      ->setTestEventCode($testEventCode);

    try {
      // Send request over to Meta.
      $request->execute();
    }
    catch (\Exception $exception) {
      $this->logger->error('Meta CAPI request failed: @message', [
        '@message' => $exception->getMessage(),
      ]);
    }
  }

  /**
   * Validates fbc (Facebook Click Cookie) value and checks if expired.
   *
   * The fbc cookie format is: fb.1.[timestamp_ms].[fbclid]
   * Example: fb.1.1554763741205.IwAR3X8K9mNpQrStUvWxYz.
   *
   * This method checks if the timestamp component is older than 90 days.
   * Meta may flag fbc values older than 90 days as "expired fbclid" in
   * Events Manager diagnostics.
   *
   * @param string $fbc
   *   The fbc cookie value to validate.
   *
   * @return bool
   *   TRUE if fbc is valid and not expired, FALSE if expired or invalid
   *   format.
   */
  protected function isValidFbc(string $fbc): bool {
    // Parse fbc format: fb.1.timestamp.fbclid.
    $parts = explode('.', $fbc);

    // Need at least 4 parts (fb, version, timestamp, fbclid).
    if (count($parts) < 4) {
      $this->logger->warning('Invalid fbc format: @fbc', ['@fbc' => $fbc]);
      return FALSE;
    }

    // Extract timestamp (in milliseconds).
    $timestamp_ms = (int) $parts[2];

    // Validate timestamp is reasonable (not zero or garbage).
    if ($timestamp_ms <= 0) {
      $this->logger->warning('Invalid fbc timestamp: @fbc', ['@fbc' => $fbc]);
      return FALSE;
    }

    // Get current time in milliseconds.
    $current_ms = (int) (microtime(TRUE) * 1000);

    // Calculate age in milliseconds.
    $age_ms = $current_ms - $timestamp_ms;

    // Define 90 days in milliseconds.
    $ninety_days_ms = 90 * 24 * 60 * 60 * 1000;

    // Check if older than 90 days.
    if ($age_ms > $ninety_days_ms) {
      $age_days = floor($age_ms / (24 * 60 * 60 * 1000));
      $this->logger->info('Filtered expired fbc cookie: @days days old', [
        '@days' => $age_days,
      ]);
      return FALSE;
    }

    return TRUE;
  }

  /**
   * Strips excluded query parameters from a URL.
   *
   * @param string $url
   *   The URL to process.
   *
   * @return string
   *   The URL with configured excluded parameters removed.
   */
  protected function stripExcludedParams(string $url): string {
    $exclude_params = $this->config->get('privacy.exclude_params');
    if (empty($exclude_params)) {
      return $url;
    }

    $parts = parse_url($url);
    if (empty($parts['query'])) {
      return $url;
    }

    parse_str($parts['query'], $params);

    // Use Drupal's built-in helper to filter parameters.
    $exclude_array = array_map('trim', explode(',', $exclude_params));
    $filtered_params = UrlHelper::filterQueryParameters($params, $exclude_array);

    // Reconstruct URL: everything before '?' + new query string.
    $base_url = strtok($url, '?');
    return $base_url . (!empty($filtered_params) ? '?' . UrlHelper::buildQuery($filtered_params) : '');
  }

}
