<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\Plugin\MetaPixelEvent;

use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\meta_pixel\Attribute\MetaPixelEvent;
use Drupal\meta_pixel\MetaPixelEventPluginBase;
use Drupal\node\NodeInterface;

/**
 * Builds ViewContent event data for nodes.
 *
 * This plugin transforms Node entities into Meta's
 * ViewContent event format.
 *
 * @see \meta_pixel_node_view()
 * @see \Drupal\meta_pixel\ViewContentLazyBuilder
 */
#[MetaPixelEvent(
  id: 'node_view_content',
  event_name: 'ViewContent',
  label: new TranslatableMarkup('View Content (Node)'),
  description: new TranslatableMarkup('Fires when viewing a node.'),
  weight: 0,
)]
class NodeViewContentEvent extends MetaPixelEventPluginBase {

  /**
   * {@inheritdoc}
   */
  public function applies(array $data): bool {
    return !empty($data['node']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildEventData(): array {
    $event_data = parent::buildEventData();
    $data = $this->getData();
    // Extract entities from context.
    $node = $data['node'] ?? NULL;
    if (!$node instanceof NodeInterface) {
      return $event_data;
    }
    $event_data['content_type'] = $node->getType();
    $event_data['content_ids'] = [$node->id()];
    $event_data['content_name'] = $node->getTitle();
    return $event_data;

  }

}
