<?php

declare(strict_types=1);

namespace Drupal\Tests\meta_pixel\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests the Meta Pixel settings form.
 *
 * @group meta_pixel
 */
class SettingsFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meta_pixel',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The test pixel ID.
   *
   * @var string
   */
  protected string $pixelId = '123456789012345';

  /**
   * A user with permission to configure meta_pixel.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $adminUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'administer meta pixel',
      'access administration pages',
    ]);
  }

  /**
   * Tests that the settings form is accessible to admin users.
   */
  public function testSettingsFormAccess(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/services/meta-pixel');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Meta Pixel');
  }

  /**
   * Tests that anonymous users cannot access the settings form.
   */
  public function testSettingsFormAccessDenied(): void {
    $this->drupalGet('/admin/config/services/meta-pixel');
    $this->assertSession()->statusCodeEquals(403);
  }

  /**
   * Tests saving the pixel ID.
   */
  public function testSavePixelId(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/services/meta-pixel');

    $this->submitForm([
      'pixel_id' => $this->pixelId,
    ], 'Save configuration');

    $this->assertSession()->pageTextContains('The configuration options have been saved.');
    $this->assertEquals(
      $this->pixelId,
      $this->config('meta_pixel.settings')->get('pixel_id')
    );
  }

  /**
   * Tests that the pixel script appears when pixel ID is configured.
   */
  public function testPixelScriptPresence(): void {
    $this->config('meta_pixel.settings')
      ->set('pixel_id', $this->pixelId)
      ->save();

    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);
    // Check that drupalSettings contains the metaPixel configuration.
    // The pixel ID appears in drupalSettings JSON.
    $this->assertSession()->responseContains('metaPixel');
    $this->assertSession()->responseContains($this->pixelId);
  }

  /**
   * Tests that the pixel script does not appear without pixel ID.
   */
  public function testNoPixelScriptWithoutId(): void {
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseNotContains('fbq');
  }

  /**
   * Tests the events configuration form.
   */
  public function testEventsFormAccess(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('/admin/config/services/meta-pixel/events');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Events');
  }

}
