<?php

declare(strict_types=1);

namespace Drupal\Tests\meta_pixel\Functional;

use Drupal\Tests\BrowserTestBase;

/**
 * Tests Meta Pixel visibility settings by role.
 *
 * @group meta_pixel
 */
class VisibilityTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'meta_pixel',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * The test pixel ID.
   *
   * @var string
   */
  protected string $pixelId = '123456789012345';

  /**
   * An authenticated user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $authenticatedUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set pixel ID.
    $this->config('meta_pixel.settings')
      ->set('pixel_id', $this->pixelId)
      ->save();

    $this->authenticatedUser = $this->drupalCreateUser([
      'access content',
    ]);
  }

  /**
   * Tests that pixel loads for anonymous users by default.
   */
  public function testPixelLoadsForAnonymous(): void {
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains($this->pixelId);
  }

  /**
   * Tests that pixel loads for authenticated users by default.
   */
  public function testPixelLoadsForAuthenticated(): void {
    $this->drupalLogin($this->authenticatedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->responseContains($this->pixelId);
  }

  /**
   * Tests excluding anonymous role from tracking.
   */
  public function testExcludeAnonymousRole(): void {
    // Set to track all roles except anonymous.
    // 'all_roles' mode means "track all EXCEPT listed roles".
    $this->config('meta_pixel.settings')
      ->set('visibility.user_role_mode', 'all_roles')
      ->set('visibility.user_role_roles', ['anonymous'])
      ->save();

    // Rebuild container to clear static caches.
    $this->rebuildContainer();

    // Anonymous should not see pixel.
    $this->drupalGet('<front>');
    $this->assertSession()->responseNotContains($this->pixelId);

    // Authenticated should still see pixel.
    $this->drupalLogin($this->authenticatedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->responseContains($this->pixelId);
  }

  /**
   * Tests excluding authenticated role from tracking.
   */
  public function testExcludeAuthenticatedRole(): void {
    // Set to track all roles except authenticated.
    // 'all_roles' mode means "track all EXCEPT listed roles".
    $this->config('meta_pixel.settings')
      ->set('visibility.user_role_mode', 'all_roles')
      ->set('visibility.user_role_roles', ['authenticated'])
      ->save();

    // Rebuild container to clear static caches.
    $this->rebuildContainer();

    // Anonymous should still see pixel.
    $this->drupalGet('<front>');
    $this->assertSession()->responseContains($this->pixelId);

    // Authenticated should not see pixel.
    $this->drupalLogin($this->authenticatedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->responseNotContains($this->pixelId);
  }

  /**
   * Tests including only specific roles.
   */
  public function testIncludeOnlyAuthenticatedRole(): void {
    // Set to only track authenticated users.
    // Any value other than 'all_roles' means "track ONLY listed roles".
    $this->config('meta_pixel.settings')
      ->set('visibility.user_role_mode', 'listed_roles')
      ->set('visibility.user_role_roles', ['authenticated'])
      ->save();

    // Rebuild container to clear static caches.
    $this->rebuildContainer();

    // Anonymous should not see pixel.
    $this->drupalGet('<front>');
    $this->assertSession()->responseNotContains($this->pixelId);

    // Authenticated should see pixel.
    $this->drupalLogin($this->authenticatedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->responseContains($this->pixelId);
  }

  /**
   * Tests including only anonymous role.
   */
  public function testIncludeOnlyAnonymousRole(): void {
    // Set to only track anonymous users.
    // Any value other than 'all_roles' means "track ONLY listed roles".
    $this->config('meta_pixel.settings')
      ->set('visibility.user_role_mode', 'listed_roles')
      ->set('visibility.user_role_roles', ['anonymous'])
      ->save();

    // Rebuild container to clear static caches.
    $this->rebuildContainer();

    // Anonymous should see pixel.
    $this->drupalGet('<front>');
    $this->assertSession()->responseContains($this->pixelId);

    // Authenticated should not see pixel.
    $this->drupalLogin($this->authenticatedUser);
    $this->drupalGet('<front>');
    $this->assertSession()->responseNotContains($this->pixelId);
  }

}
