<?php

declare(strict_types=1);

namespace Drupal\Tests\meta_pixel\Kernel;

use Drupal\meta_pixel\MetaPixelEvent;

/**
 * Helper trait to assert Meta Pixel events.
 */
trait AssertMetaPixelTrait {

  /**
   * Asserts that browser events contain expected events.
   *
   * @param array $expected_events
   *   Array of expected events with 'name' and optionally 'data' keys.
   */
  protected function assertBrowserEvents(array $expected_events): void {
    $collector = $this->container->get('meta_pixel.event_collector');
    $events = $collector->getBrowserEvents();

    if ($expected_events === []) {
      self::assertEmpty($events, 'Expected no browser events but found some.');
      return;
    }

    self::assertCount(count($expected_events), $events, 'Browser event count mismatch.');

    $events = array_values($events);
    foreach ($expected_events as $index => $expected) {
      /** @var \Drupal\meta_pixel\MetaPixelEvent $event */
      $event = $events[$index];
      self::assertInstanceOf(MetaPixelEvent::class, $event);
      self::assertEquals($expected['name'], $event->getName(), "Event name mismatch at index $index.");

      if (isset($expected['data'])) {
        $event_data = $event->getEventData();
        foreach ($expected['data'] as $key => $value) {
          self::assertArrayHasKey($key, $event_data, "Missing key '$key' in event data.");
          self::assertEquals($value, $event_data[$key], "Event data mismatch for key '$key'.");
        }
      }
    }
  }

  /**
   * Asserts that CAPI events contain expected events.
   *
   * @param array $expected_events
   *   Array of expected events with 'name' and optionally 'data' keys.
   */
  protected function assertCapiEvents(array $expected_events): void {
    $collector = $this->container->get('meta_pixel.event_collector');
    $events = $collector->getCapiEvents();

    if ($expected_events === []) {
      self::assertEmpty($events, 'Expected no CAPI events but found some.');
      return;
    }

    self::assertCount(count($expected_events), $events, 'CAPI event count mismatch.');

    $events = array_values($events);
    foreach ($expected_events as $index => $expected) {
      /** @var \Drupal\meta_pixel\MetaPixelEvent $event */
      $event = $events[$index];
      self::assertInstanceOf(MetaPixelEvent::class, $event);
      self::assertEquals($expected['name'], $event->getName(), "Event name mismatch at index $index.");

      if (isset($expected['data'])) {
        $event_data = $event->getEventData();
        foreach ($expected['data'] as $key => $value) {
          self::assertArrayHasKey($key, $event_data, "Missing key '$key' in event data.");
          self::assertEquals($value, $event_data[$key], "Event data mismatch for key '$key'.");
        }
      }
    }
  }

  /**
   * Asserts that an event has the expected name.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The event to check.
   * @param string $expected_name
   *   The expected event name.
   */
  protected function assertEventName(MetaPixelEvent $event, string $expected_name): void {
    self::assertEquals($expected_name, $event->getName());
  }

  /**
   * Asserts that an event has a valid event ID.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The event to check.
   */
  protected function assertEventHasId(MetaPixelEvent $event): void {
    $event_id = $event->getEventId();
    self::assertNotEmpty($event_id, 'Event should have an event_id.');
    self::assertIsString($event_id);
  }

  /**
   * Asserts that an event has expected data values.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The event to check.
   * @param array $expected_data
   *   Key-value pairs to check in event data.
   */
  protected function assertEventData(MetaPixelEvent $event, array $expected_data): void {
    $event_data = $event->getEventData();
    foreach ($expected_data as $key => $value) {
      self::assertArrayHasKey($key, $event_data, "Missing key '$key' in event data.");
      self::assertEquals($value, $event_data[$key], "Event data mismatch for key '$key'.");
    }
  }

}
