<?php

declare(strict_types=1);

namespace Drupal\Tests\meta_pixel\Kernel;

/**
 * Tests the CustomEvent plugin.
 *
 * @group meta_pixel
 */
final class CustomEventTest extends MetaPixelTestCase {

  /**
   * Tests custom event with event_name override.
   */
  public function testCustomEventWithEventName(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
      'content_name' => 'Free Download',
      'content_category' => 'File Downloads',
      'offer_id' => 'OFFER_123',
    ]);

    self::assertNotNull($event);
    self::assertEquals('Lead', $event->getName());

    $data = $event->getEventData();
    self::assertEquals('Free Download', $data['content_name']);
    self::assertEquals('File Downloads', $data['content_category']);
    self::assertEquals('OFFER_123', $data['offer_id']);
  }

  /**
   * Tests custom event with content_ids array.
   */
  public function testCustomEventWithContentIds(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'ViewContent',
      'content_ids' => ['SKU-001', 'SKU-002'],
      'content_type' => 'product',
    ]);

    self::assertNotNull($event);

    $data = $event->getEventData();
    self::assertEquals(['SKU-001', 'SKU-002'], $data['content_ids']);
    self::assertEquals('product', $data['content_type']);
  }

  /**
   * Tests custom event with value and currency.
   */
  public function testCustomEventWithValue(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Purchase',
      'value' => 99.99,
      'currency' => 'USD',
    ]);

    self::assertNotNull($event);

    $data = $event->getEventData();
    self::assertEquals(99.99, $data['value']);
    self::assertEquals('USD', $data['currency']);
  }

  /**
   * Tests custom event with predicted LTV.
   */
  public function testCustomEventWithPredictedLtv(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
      'predicted_ltv' => 299.00,
    ]);

    self::assertNotNull($event);

    $data = $event->getEventData();
    self::assertEquals(299.00, $data['predicted_ltv']);
  }

  /**
   * Tests that custom event without event_name uses plugin default.
   */
  public function testCustomEventDefaultName(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'content_name' => 'Test',
    ]);

    self::assertNotNull($event);
    // Default from plugin annotation.
    self::assertEquals('CustomEvent', $event->getName());
  }

  /**
   * Tests custom event session storage for redirects.
   */
  public function testCustomEventWithSession(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();

    // Add event with session storage.
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
    ], TRUE);

    self::assertNotNull($event);

    // Event should have session cache context.
    $contexts = $event->getCacheContexts();
    self::assertContains('session', $contexts);
  }

}
