<?php

declare(strict_types=1);

namespace Drupal\Tests\meta_pixel\Kernel;

use Drupal\Tests\user\Traits\UserCreationTrait;
use Drupal\user\Entity\User;

/**
 * Tests the Meta Pixel EventCollector service.
 *
 * @group meta_pixel
 * @coversDefaultClass \Drupal\meta_pixel\EventCollector
 */
final class EventCollectorTest extends MetaPixelTestCase {

  use UserCreationTrait;

  /**
   * Tests that the event collector service exists.
   */
  public function testServiceExists(): void {
    $collector = $this->getEventCollector();
    self::assertNotNull($collector);
    self::assertInstanceOf('Drupal\meta_pixel\EventCollectorInterface', $collector);
  }

  /**
   * Tests adding a custom event for browser tracking.
   *
   * @covers ::addEvent
   * @covers ::getBrowserEvents
   */
  public function testAddCustomEventBrowser(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
      'content_name' => 'Test Content',
      'content_category' => 'Test Category',
    ]);

    self::assertNotNull($event);
    self::assertEquals('Lead', $event->getName());
    $this->assertEventHasId($event);

    // Check browser events queue.
    $this->assertBrowserEvents([
      [
        'name' => 'Lead',
        'data' => [
          'content_name' => 'Test Content',
          'content_category' => 'Test Category',
        ],
      ],
    ]);
  }

  /**
   * Tests adding a custom event for CAPI tracking.
   *
   * @covers ::addEvent
   * @covers ::getCapiEvents
   */
  public function testAddCustomEventCapi(): void {
    $this->setPixelId();
    $this->enableCapiEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
      'content_name' => 'Test Content',
    ]);

    self::assertNotNull($event);

    // Check CAPI events queue.
    $this->assertCapiEvents([
      [
        'name' => 'Lead',
        'data' => [
          'content_name' => 'Test Content',
        ],
      ],
    ]);
  }

  /**
   * Tests that events are not added when pixel ID is missing.
   *
   * @covers ::addEvent
   */
  public function testNoEventWithoutPixelId(): void {
    // Don't set pixel ID.
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
    ]);

    self::assertNull($event);
    $this->assertBrowserEvents([]);
  }

  /**
   * Tests that events are not added when plugin is disabled.
   *
   * @covers ::addEvent
   */
  public function testNoEventWhenDisabled(): void {
    $this->setPixelId();
    // Don't enable any events.
    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
    ]);

    self::assertNull($event);
    $this->assertBrowserEvents([]);
  }

  /**
   * Tests isBrowserEnabled method.
   *
   * @covers ::isBrowserEnabled
   */
  public function testIsBrowserEnabled(): void {
    $this->setPixelId();
    $collector = $this->getEventCollector();

    self::assertFalse($collector->isBrowserEnabled('custom_event'));

    $this->enableBrowserEvent('custom_event');
    self::assertTrue($collector->isBrowserEnabled('custom_event'));
  }

  /**
   * Tests isCapiEnabled method.
   *
   * @covers ::isCapiEnabled
   */
  public function testIsCapiEnabled(): void {
    $this->setPixelId();
    $collector = $this->getEventCollector();

    self::assertFalse($collector->isCapiEnabled('custom_event'));

    $this->enableCapiEvent('custom_event');
    self::assertTrue($collector->isCapiEnabled('custom_event'));
  }

  /**
   * Tests isAnyEnabled method.
   *
   * @covers ::isAnyEnabled
   */
  public function testIsAnyEnabled(): void {
    $this->setPixelId();
    $collector = $this->getEventCollector();

    self::assertFalse($collector->isAnyEnabled('custom_event'));

    $this->enableBrowserEvent('custom_event');
    self::assertTrue($collector->isAnyEnabled('custom_event'));
  }

  /**
   * Tests event deduplication.
   *
   * @covers ::getBrowserEvents
   */
  public function testEventDeduplication(): void {
    $this->setPixelId();
    $this->enableBrowserEvent('custom_event');

    $collector = $this->getEventCollector();

    // Add same event twice - should be deduplicated.
    $event1 = $collector->addEvent('custom_event', ['event_name' => 'Lead']);
    $event2 = $collector->addEvent('custom_event', ['event_name' => 'Lead']);

    // Different event IDs means both should be present.
    self::assertNotEquals($event1->getEventId(), $event2->getEventId());

    $events = $collector->getBrowserEvents();
    self::assertCount(2, $events);
  }

  /**
   * Tests buildUserData with current user.
   *
   * @covers ::buildUserData
   */
  public function testBuildUserDataWithCurrentUser(): void {
    $this->setPixelId();

    // Create a test user.
    $user = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
      'status' => 1,
    ]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);

    $collector = $this->getEventCollector();
    $context = [];
    $user_data = $collector->buildUserData([], $context);

    self::assertArrayHasKey('external_id', $user_data);
    self::assertContains('u_' . $user->id(), $user_data['external_id']);
    self::assertArrayHasKey('email', $user_data);
    self::assertContains('test@example.com', $user_data['email']);
  }

  /**
   * Tests that anonymous users don't have user data.
   *
   * @covers ::buildUserData
   */
  public function testBuildUserDataAnonymous(): void {
    $this->setPixelId();

    $collector = $this->getEventCollector();
    $context = [];
    $user_data = $collector->buildUserData([], $context);

    self::assertEmpty($user_data);
  }

  /**
   * Tests prepareCapiUserData flattening.
   *
   * @covers ::prepareCapiUserData
   */
  public function testPrepareCapiUserData(): void {
    $this->setPixelId();

    $collector = $this->getEventCollector();

    $user_data = [
      'email' => ['test1@example.com', 'test2@example.com'],
      'external_id' => ['u_123'],
    ];

    $result = $collector->prepareCapiUserData($user_data);

    // Multiple emails should use plural key.
    self::assertArrayHasKey('emails', $result);
    self::assertCount(2, $result['emails']);

    // Single external_id should use singular key.
    self::assertArrayHasKey('external_id', $result);
    self::assertEquals('u_123', $result['external_id']);
  }

  /**
   * Tests that CAPI events contain user data from authenticated user.
   *
   * @covers ::addEvent
   * @covers ::getCapiEvents
   */
  public function testCapiEventContainsUserData(): void {
    $this->setPixelId();
    $this->setCapiAccessToken();
    $this->enableCapiEvent('custom_event');

    // Create and set authenticated user.
    $user = User::create([
      'name' => 'capi_user',
      'mail' => 'capi@example.com',
      'status' => 1,
    ]);
    $user->save();
    $this->container->get('current_user')->setAccount($user);

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'Lead',
    ]);

    self::assertNotNull($event);

    // Verify the event has user data attached.
    $user_data = $event->getUserData();
    self::assertNotEmpty($user_data, 'Event should have user data attached');
    self::assertArrayHasKey('external_id', $user_data);
    self::assertArrayHasKey('email', $user_data);
    self::assertContains('u_' . $user->id(), $user_data['external_id']);
    self::assertContains('capi@example.com', $user_data['email']);

    // Verify CAPI queue has the event.
    $capi_events = $collector->getCapiEvents();
    self::assertCount(1, $capi_events);
  }

  /**
   * Tests that CAPI events for anonymous users have no user data.
   *
   * @covers ::addEvent
   */
  public function testCapiEventAnonymousNoUserData(): void {
    $this->setPixelId();
    $this->setCapiAccessToken();
    $this->enableCapiEvent('custom_event');

    $collector = $this->getEventCollector();
    $event = $collector->addEvent('custom_event', [
      'event_name' => 'PageView',
    ]);

    self::assertNotNull($event);

    // Anonymous user should have no user data.
    $user_data = $event->getUserData();
    self::assertEmpty($user_data, 'Anonymous user should have no user data');
  }

  /**
   * Tests that user data from context takes precedence.
   *
   * When a user entity is passed in context, their data should be used
   * even if a different user is logged in.
   *
   * @covers ::buildUserData
   */
  public function testBuildUserDataFromContext(): void {
    $this->setPixelId();

    // Create two users.
    $current_user = User::create([
      'name' => 'currentuser',
      'mail' => 'current@example.com',
      'status' => 1,
    ]);
    $current_user->save();

    $context_user = User::create([
      'name' => 'context_user',
      'mail' => 'context@example.com',
      'status' => 1,
    ]);
    $context_user->save();

    // Set current user.
    $this->container->get('current_user')->setAccount($current_user);

    $collector = $this->getEventCollector();

    // Pass different user in context.
    $context = ['user' => $context_user];
    $user_data = $collector->buildUserData([], $context);

    // Should have current user's data. The 'user' key in context is set
    // BY buildUserData, not read from it.
    self::assertContains('current@example.com', $user_data['email']);
  }

  /**
   * Tests email deduplication in user data.
   *
   * @covers ::prepareCapiUserData
   */
  public function testPrepareCapiUserDataDeduplicatesEmails(): void {
    $this->setPixelId();

    $collector = $this->getEventCollector();

    // Same email appears multiple times.
    $user_data = [
      'email' => [
        'duplicate@example.com',
        'unique@example.com',
        'duplicate@example.com',
      ],
    ];

    $result = $collector->prepareCapiUserData($user_data);

    // Should deduplicate.
    self::assertArrayHasKey('emails', $result);
    self::assertCount(2, $result['emails']);
    self::assertContains('duplicate@example.com', $result['emails']);
    self::assertContains('unique@example.com', $result['emails']);
  }

  /**
   * Tests profile data flattening in prepareCapiUserData.
   *
   * @covers ::prepareCapiUserData
   */
  public function testPrepareCapiUserDataWithProfiles(): void {
    $this->setPixelId();

    $collector = $this->getEventCollector();

    $user_data = [
      'email' => ['user@example.com'],
      'external_id' => ['u_123'],
      'profiles' => [
        [
          'first_name' => 'John',
          'last_name' => 'Doe',
          'city' => 'New York',
          'zip_code' => '10001',
        ],
        [
          'first_name' => 'Jane',
          'last_name' => 'Doe',
          'city' => 'Boston',
        ],
      ],
    ];

    $result = $collector->prepareCapiUserData($user_data);

    // Should flatten profiles.
    self::assertArrayHasKey('first_names', $result);
    self::assertCount(2, $result['first_names']);
    self::assertContains('John', $result['first_names']);
    self::assertContains('Jane', $result['first_names']);

    // Last name appears twice but should be deduplicated.
    self::assertArrayHasKey('last_name', $result);
    self::assertEquals('Doe', $result['last_name']);

    // Cities should be plural.
    self::assertArrayHasKey('cities', $result);
    self::assertCount(2, $result['cities']);
  }

}
