<?php

declare(strict_types=1);

namespace Drupal\Tests\meta_pixel\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;

/**
 * Base class for meta_pixel kernel tests.
 */
abstract class MetaPixelTestCase extends KernelTestBase {

  use AssertMetaPixelTrait;

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'system',
    'user',
    'meta_pixel',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();
    $this->installEntitySchema('user');
    $this->installConfig(['system', 'user', 'meta_pixel']);
  }

  /**
   * Configures a test pixel ID.
   *
   * @param string $pixel_id
   *   The pixel ID to configure.
   */
  protected function setPixelId(string $pixel_id = '123456789'): void {
    $this->config('meta_pixel.settings')
      ->set('pixel_id', $pixel_id)
      ->save();
  }

  /**
   * Enables an event plugin for browser tracking.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   */
  protected function enableBrowserEvent(string $plugin_id): void {
    $this->config('meta_pixel.events')
      ->set('browser_enabled.' . $plugin_id, TRUE)
      ->save();
  }

  /**
   * Sets the CAPI access token.
   *
   * @param string $access_token
   *   The access token.
   */
  protected function setCapiAccessToken(string $access_token = 'test_token'): void {
    $this->config('meta_pixel.settings')
      ->set('capi.access_token', $access_token)
      ->save();
  }

  /**
   * Enables an event plugin for CAPI tracking.
   *
   * @param string $plugin_id
   *   The event plugin ID.
   * @param string $access_token
   *   Optional access token.
   */
  protected function enableCapiEvent(string $plugin_id, string $access_token = 'test_token'): void {
    $this->setCapiAccessToken($access_token);
    $this->config('meta_pixel.events')
      ->set('capi_enabled.' . $plugin_id, TRUE)
      ->save();
  }

  /**
   * Gets the event collector service.
   *
   * @return \Drupal\meta_pixel\EventCollectorInterface
   *   The event collector.
   */
  protected function getEventCollector() {
    return $this->container->get('meta_pixel.event_collector');
  }

  /**
   * Sends a request to drupal kernel and builds the response.
   *
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   Request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   Response.
   *
   * @throws \Exception
   */
  protected function doRequest(Request $request): Response {
    $response = $this->container->get('http_kernel')->handle($request);
    $content = $response->getContent();
    self::assertNotFalse($content);
    $this->setRawContent($content);
    return $response;
  }

}
