/**
 * @file
 * meta_pixel.js
 *
 * Defines the behavior of the meta pixel module.
 */

(function (Drupal, drupalSettings, once) {

  'use strict';

  // Initialize Drupal.metaPixel namespace.
  Drupal.metaPixel = Drupal.metaPixel || {};

  /**
   * Generates a unique event ID for deduplication.
   *
   * @param {string} eventName
   *   The event name (e.g., 'PageView', 'AddToCart').
   *
   * @return {string}
   *   Unique event ID in format: eventname-timestamp-random.
   */
  Drupal.metaPixel.generateEventId = function (eventName) {
    var timestamp = Math.floor(Date.now() / 1000);
    var random = Math.random().toString(36).substring(2, 15);
    return eventName.toLowerCase() + '-' + timestamp + '-' + random;
  };

  Drupal.metaPixel.removeUrlParams = function () {
    const url = window.location.href;
    try {
      // Use the URL API for modern, reliable parsing
      const urlObj = new URL(window.location.href);
      const searchParams = urlObj.searchParams;

      if (drupalSettings.metaPixel.exclude_params) {
        // Iterate through the specified parameters to remove
        drupalSettings.metaPixel.exclude_params.forEach(param => {
          if (searchParams.has(param)) {
            searchParams.delete(param);
          }
        });
        // Reconstruct the URL without the removed parameters
        return urlObj.origin + urlObj.pathname + urlObj.search;
      }
      return url;
    } catch (error) {
      // Return original URL if there's an error
      return url;
    }

  };

  Drupal.metaPixel.initTracking = function () {
    if (drupalSettings.metaPixel.pixelId) {
      // Define Drupal.metaPixel.fb_disable as a dynamic condition to
      // disable FB Pixel at runtime.
      // This is helpful for GDPR compliance module integration
      // and works even with static caching mechanisms like boost module.
      Drupal.metaPixel.fb_disable = false;
    }

    // Define Opt-out conditions check.
    if (drupalSettings.metaPixel.disable_advanced_matching) {
      // Facebook Pixel Opt-Out Code.
      window.fbOptout = function (reload = 0) {
        reload = (typeof reload !== 'undefined') ? reload : 0;
        var optoutWindowDisableKey = 'fb-disable';
        document.cookie = optoutWindowDisableKey + '=true; expires=Thu, 31 Dec 2999 23:59:59 UTC; path=/';
        window[optoutWindowDisableKey] = true;
        if (reload) {
          location.reload();
        }
      };
      if (document.cookie.indexOf('fb-disable=true') > -1) {
        window['fb-disable'] = true;
      }
      // End Facebook Pixel Opt-Out Code.
      Drupal.metaPixel.fb_disable = Drupal.metaPixel.fb_disable || window['fb-disable'] == true;
    }

    // Define eu_cookie_compliance conditions check (https://www.drupal.org/project/eu_cookie_compliance).
    if (drupalSettings.metaPixel.eu_cookie_compliance) {
      if (typeof Drupal.eu_cookie_compliance === 'undefined') {
        console.warn('meta_pixel: meta_pixel eu_cookie_compliance integration option is enabled, but eu_cookie_compliance javascripts seem to be loaded after meta_pixel, which may break functionality.');
      }
      var eccHasAgreed = (typeof Drupal.eu_cookie_compliance !== 'undefined' && Drupal.eu_cookie_compliance.hasAgreed());
      Drupal.metaPixel.fb_disable = Drupal.metaPixel.fb_disable || !eccHasAgreed;
    }

    // Define Do-not-track conditions check (see https://www.w3.org/TR/tracking-dnt/).
    if (drupalSettings.metaPixel.respect_dnt) {
      const DNT = (typeof navigator.doNotTrack !== 'undefined' && (navigator.doNotTrack === 'yes' || navigator.doNotTrack == 1)) || (typeof navigator.msDoNotTrack !== 'undefined' && navigator.msDoNotTrack == 1) || (typeof window.doNotTrack !== 'undefined' && window.doNotTrack == 1);
      // If eccHasAgreed is true, it overrides DNT because eu_cookie_compliance contains a setting for opt-in with DNT:
      // "Automatic. Respect the DNT (Do not track) setting in the browser, if present. Uses opt-in when DNT is 1 or not set, and consent by default when DNT is 0."
      Drupal.metaPixel.fb_disable = Drupal.metaPixel.fb_disable || (DNT && (typeof eccHasAgreed == 'undefined' || !eccHasAgreed));
    }

    if (!Drupal.metaPixel.fb_disable) {
      let elements = once('meta_pixel_pageload_tracking', 'body');
      elements.forEach(function () {
        !function (f, b, e, v, n, t, s) {
          if (f.fbq) {
            return;
          }
          n = f.fbq = function () {
            n.callMethod ?
              n.callMethod.apply(n, arguments) : n.queue.push(arguments);
          };
          if (!f._fbq) {
            f._fbq = n;
          }
          n.push = n;
          n.loaded = !0;
          n.version = '2.0';
          n.queue = [];
          t = b.createElement(e);
          t.async = !0;
          t.src = v;
          s = b.getElementsByTagName(e)[0];
          s.parentNode.insertBefore(t, s);
        }(window, document, 'script',
          'https://connect.facebook.net/en_US/fbevents.js');

        const userData = drupalSettings.metaPixel.userData || {};
        const pixelId = drupalSettings.metaPixel.pixelId;

        // Disable push state to prevent Facebook from automatically tracking
        // URL changes (query string changes from AJAX).
        // This prevents the pixel from re-initializing when product variations
        // change URL which was breaking tracking.
        fbq.disablePushState = true;

        let initData = {};
        if (!drupalSettings.metaPixel.disable_advanced_matching && userData) {
          initData = userData;
        }

        if (drupalSettings.metaPixel.testCodeId) {
          //initData.testEventCode = drupalSettings.metaPixel.testCodeId;
          fbq('set', 'testEventCode', drupalSettings.metaPixel.testCodeId, pixelId);
        }
        fbq('init', pixelId, initData);

        // Generate event_id for PageView.
        const pageViewEventId = Drupal.metaPixel.generateEventId('pageview');
        const sourceUrl = Drupal.metaPixel.removeUrlParams();
        fbq('set', 'event_source_url', sourceUrl, pixelId);
        if (drupalSettings.metaPixel.pageview_browser) {
          // Fire PageView to browser pixel with event_id.
          fbq('track', 'PageView', {}, { eventID: pageViewEventId });
        }

        // If CAPI is enabled for PageView, send to server with same event_id.
        if (drupalSettings.metaPixel.pageview_capi) {
          const params = new URLSearchParams({
            event_id: pageViewEventId,
            url: sourceUrl,
          });

          fetch('/ajax/pageview?' + params.toString());
        }
      });
    }
  };

  Drupal.behaviors.meta_pixel = {
    attach(context) {
      once('meta_pixel_behavior', 'html').forEach(function () {
        Drupal.metaPixel.initTracking();
        document.addEventListener('eu_cookie_compliance.changeStatus', function () {
          Drupal.metaPixel.initTracking();
        });
      });

      // Fire any queued events
      if (!Drupal.metaPixel.fb_disable && drupalSettings.metaPixel && drupalSettings.metaPixel.events) {
        const sourceUrl = Drupal.metaPixel.removeUrlParams();
        for (const [id, event] of Object.entries(drupalSettings.metaPixel.events)) {
          const eventName = event.event_name;
          const eventData = event.event_data || {};
          const options = event.event_id ? { eventID: event.event_id } : {};
          if (drupalSettings.metaPixel.debug) {
            console.log(eventName, eventData, options);
          }
          fbq('track', eventName, eventData, options);
        }
        drupalSettings.metaPixel.events = [];
      }

    },
  };

}(Drupal, drupalSettings, once));
