<?php

declare(strict_types=1);

namespace Drupal\meta_pixel_commerce\EventSubscriber;

use Drupal\Core\Session\AccountProxyInterface;
use Drupal\meta_pixel\EventCollector;
use Drupal\state_machine\Event\WorkflowTransitionEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Tracks Purchase events for Meta Pixel.
 */
class PurchaseSubscriber implements EventSubscriberInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  protected EventCollector $eventCollector;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * Constructs a PurchaseSubscriber object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector service.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   */
  public function __construct(
    EventCollector $eventCollector,
    AccountProxyInterface $currentUser
  ) {
    $this->eventCollector = $eventCollector;
    $this->currentUser = $currentUser;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      'commerce_order.place.post_transition' => 'onOrderPlace',
    ];
  }

  /**
   * Fires Purchase event when order is placed (checkout completion).
   *
   * This fires on the thank you page after successful checkout.
   * Uses order_id as context for event_id generation to ensure
   * browser and server events match for proper deduplication.
   *
   * @param \Drupal\state_machine\Event\WorkflowTransitionEvent $event
   *   The workflow transition event.
   */
  public function onOrderPlace(WorkflowTransitionEvent $event): void {
    /** @var \Drupal\commerce_order\Entity\OrderInterface $order */
    $order = $event->getEntity();

    if (!$order) {
      return;
    }

    // If the acting user is not the customer, an admin is placing the order.
    if ($order->getCustomerId() != $this->currentUser->id()) {
      return;
    }

    // Add context for caching and alter hook.
    $data['order'] = $order;

    $this->eventCollector->addEvent('commerce_purchase', $data, TRUE);
  }

}
