<?php

declare(strict_types=1);

namespace Drupal\meta_pixel_commerce\EventSubscriber;

use Drupal\commerce_product\Event\ProductEvents;
use Drupal\commerce_product\Event\ProductVariationAjaxChangeEvent;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\meta_pixel\EventCollector;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Fires ViewContent event when product variation changes.
 */
class VariationSubscriber implements EventSubscriberInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  private EventCollector $eventCollector;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  private AccountProxyInterface $currentUser;

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  private RequestStack $requestStack;

  /**
   * Constructs a CartSubscriber object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   */
  public function __construct(
    EventCollector $eventCollector,
    AccountProxyInterface $currentUser,
    RequestStack $requestStack
  ) {
    $this->eventCollector = $eventCollector;
    $this->currentUser = $currentUser;
    $this->requestStack = $requestStack;
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      ProductEvents::PRODUCT_VARIATION_AJAX_CHANGE => 'onVariationChange',
    ];
  }

  /**
   * Fires ViewContent event when product variation changes.
   *
   * @param \Drupal\commerce_product\Event\ProductVariationAjaxChangeEvent $event
   *   The variation change event.
   */
  public function onVariationChange(ProductVariationAjaxChangeEvent $event): void {
    $variation = $event->getProductVariation();
    if (!$variation) {
      return;
    }

    $product = $variation->getProduct();
    if (!$product) {
      return;
    }

    $data['product'] = $product;
    $data['product_variation'] = $variation;

    $this->eventCollector->addEvent('commerce_view_content', $data);
  }

}
