<?php

declare(strict_types=1);

namespace Drupal\meta_pixel_commerce\Plugin\MetaPixelEvent;

use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\meta_pixel\MetaPixelEventPluginBase;
use Drupal\meta_pixel\MetaPixelEventPluginInterface;

/**
 * Base class for Commerce-specific Meta Pixel event plugins.
 *
 * Provides helper methods for building event data from Commerce entities
 * like orders, order items, and product variations.
 */
abstract class CommerceMetaPixelEventPluginBase extends MetaPixelEventPluginBase implements MetaPixelEventPluginInterface, ContainerFactoryPluginInterface {

  /**
   * Gets product SKU from variation entity.
   *
   * Helper method to extract SKU using the appropriate method
   * or falling back to entity ID.
   *
   * @param mixed $variation
   *   Product variation entity.
   *
   * @return string
   *   The SKU or entity ID as string.
   */
  protected function getSku($variation): string {
    if (method_exists($variation, 'getSku')) {
      return $variation->getSku();
    }
    if (method_exists($variation, 'id')) {
      return (string) $variation->id();
    }
    return '';
  }

  /**
   * Builds event data from an order entity.
   *
   * This is a helper method for checkout-related events (InitiateCheckout,
   * AddPaymentInfo, Purchase) that all share the same data structure.
   *
   * @param mixed $order
   *   The order entity.
   *
   * @return array
   *   Meta-formatted event data with order items, totals, etc.
   */
  protected function buildOrderEventData($order): array {
    if (!$order) {
      return [];
    }

    $content_ids = [];
    $contents = [];

    foreach ($order->getItems() as $order_item) {
      $purchased_entity = $order_item->getPurchasedEntity();
      if (!$purchased_entity) {
        continue;
      }

      $sku = $this->getSku($purchased_entity);
      $content_ids[] = $sku;

      // Get adjusted unit price (after discounts).
      $adjusted_unit_price = $order_item->getAdjustedUnitPrice();

      $contents[] = [
        'id' => $sku,
        'title' => $order_item->getTitle(),
        'quantity' => (int) $order_item->getQuantity(),
        'item_price' => (float) $adjusted_unit_price->getNumber(),
      ];
    }

    $total_price = $order->getTotalPrice();
    if (!$total_price) {
      return [];
    }

    return [
      'content_type' => 'product',
      'content_ids' => $content_ids,
      'currency' => $total_price->getCurrencyCode(),
      'value' => (float) $total_price->getNumber(),
      'contents' => $contents,
      'num_items' => count($order->getItems()),
      'order_id' => $order->id(),
    ];
  }

}
