<?php

declare(strict_types=1);

namespace Drupal\meta_pixel_commerce\Plugin\MetaPixelEvent;

use Drupal\commerce_product\Entity\ProductVariationInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\meta_pixel\Attribute\MetaPixelEvent;

/**
 * Builds ViewContent event data for product variations.
 *
 * This plugin transforms Commerce product entities into Meta's
 * ViewContent event format. It handles both initial page views
 * and AJAX variation changes.
 *
 * @see \meta_pixel_commerce_commerce_product_view()
 * @see \Drupal\meta_pixel\ViewContentLazyBuilder
 * @see \Drupal\meta_pixel_commerce\EventSubscriber\VariationSubscriber
 */
#[MetaPixelEvent(
  id: 'commerce_view_content',
  event_name: 'ViewContent',
  label: new TranslatableMarkup('View Content (Commerce)'),
  description: new TranslatableMarkup('Fires when viewing a product page or changing variations.'),
  weight: 0,
)]
class ViewContentEvent extends CommerceMetaPixelEventPluginBase {

  /**
   * {@inheritdoc}
   */
  public function applies(array $data): bool {
    return !empty($data['product_variation']);
  }

  /**
   * {@inheritdoc}
   */
  public function buildEventData(): array {
    $event_data = parent::buildEventData();
    $data = $this->getData();

    // Extract entities from context.
    $variation = $data['product_variation'] ?? NULL;
    $product = $data['product'] ?? NULL;

    if (!$variation instanceof ProductVariationInterface) {
      return [];
    }

    // Load product if not provided.
    if (!$product && method_exists($variation, 'getProduct')) {
      $product = $variation->getProduct();
    }

    // Get price.
    $price = $variation->getPrice();
    if (empty($price)) {
      return [];
    }

    // Get SKU.
    $sku = $this->getSku($variation);

    // Build Meta-formatted event data.
    $event_data['content_type'] = 'product';
    $event_data['content_ids'] = [$sku];
    $event_data['content_name'] = $variation->getTitle();
    $event_data['currency'] = $price->getCurrencyCode();
    $event_data['value'] = $price->getNumber();
    $event_data['contents'] = [
      [
        'id' => $sku,
        'quantity' => 1,
        'item_price' => $price->getNumber(),
      ],
    ];

    return $event_data;
  }

}
