<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager;
use Drupal\user\UserInterface;
use FacebookAds\Object\ServerSide\Content;
use FacebookAds\Object\ServerSide\Event;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Session\SessionInterface;

/**
 * Central event collection service for Meta Pixel tracking.
 *
 * This service coordinates dual-channel tracking across browser-side pixel
 * (fbq) and server-side Conversions API (CAPI). It ensures both channels
 * use matching event IDs for proper deduplication, preventing Meta from
 * double-counting conversions.
 *
 * Architecture:
 * - Event plugins build event-specific data from Drupal entities
 * - EventCollector enriches events with user identification data
 * - Alter hooks allow site-specific customizations
 * - Browser events queue for JavaScript via drupalSettings
 * - CAPI events queue for server-side transmission during kernel.terminate
 *
 * Key features:
 * - Automatic event ID generation for deduplication
 * - User data collection for Event Match Quality
 * - Session-based storage for POST-redirect-GET patterns
 * - Cache metadata tracking from source entities
 */
class EventCollector implements EventCollectorInterface {

  /**
   * The module handler service.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The current user service.
   *
   * @var \Drupal\Core\Session\AccountProxyInterface
   */
  protected AccountProxyInterface $currentUser;

  /**
   * The logger channel for Meta Pixel operations.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  /**
   * The session service.
   *
   * @var \Symfony\Component\HttpFoundation\Session\SessionInterface
   */
  private SessionInterface $session;

  /**
   * The configuration factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The Meta Pixel event plugin manager.
   *
   * @var \Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager
   */
  protected MetaPixelEventPluginManager $eventPluginManager;

  /**
   * The Meta Conversions API client service.
   *
   * @var \Drupal\meta_pixel\MetaCapiClient
   */
  protected MetaCapiClient $metaCapiClient;

  /**
   * Constructs an EventCollector object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $moduleHandler
   *   The module handler.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Session\AccountProxyInterface $currentUser
   *   The current user.
   * @param \Drupal\Core\Logger\LoggerChannelFactoryInterface $loggerFactory
   *   The logger factory.
   * @param \Symfony\Component\HttpFoundation\Session\SessionInterface $session
   *   The session service.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $configFactory
   *   The config factory.
   * @param \Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager $eventPluginManager
   *   The Meta Pixel event plugin manager.
   * @param \Drupal\meta_pixel\MetaCapiClient $metaCapiClient
   *   The Meta CAPI client service.
   */
  public function __construct(
    EntityTypeManagerInterface $entityTypeManager,
    ModuleHandlerInterface $moduleHandler,
    RequestStack $requestStack,
    AccountProxyInterface $currentUser,
    LoggerChannelFactoryInterface $loggerFactory,
    SessionInterface $session,
    ConfigFactoryInterface $configFactory,
    MetaPixelEventPluginManager $eventPluginManager,
    MetaCapiClient $metaCapiClient,
  ) {
    $this->entityTypeManager = $entityTypeManager;
    $this->moduleHandler = $moduleHandler;
    $this->requestStack = $requestStack;
    $this->currentUser = $currentUser;
    $this->logger = $loggerFactory->get('meta_pixel');
    $this->session = $session;
    $this->configFactory = $configFactory;
    $this->eventPluginManager = $eventPluginManager;
    $this->metaCapiClient = $metaCapiClient;
  }

  /**
   * {@inheritdoc}
   */
  public function addEvent(string $plugin_id, array $data = [], bool $use_session = FALSE): MetaPixelEvent|null {
    if (!_meta_pixel_visibility_roles($this->currentUser)) {
      return NULL;
    }

    $plugin = $this->eventPluginManager->getApplicablePlugin($plugin_id, $data);
    if (!$plugin) {
      return NULL;
    }

    $capi_enabled = $this->isCapiEnabled($plugin_id);
    $browser_enabled = $this->isBrowserEnabled($plugin_id);

    // If neither tracking method is enabled, bail early.
    if (!$capi_enabled && !$browser_enabled) {
      return NULL;
    }

    $event_name = $plugin->getEventName();
    $event_data = $plugin->buildEventData();
    $context = $data;

    // Build user data for Event Match Quality and advanced matching.
    $user_data = $this->buildUserData($event_data, $context);

    // Allow other modules to alter the event and user data.
    $context['event_name'] = $event_name;
    $this->moduleHandler->alter('meta_pixel_event_data', $event_data, $user_data, $context);

    $event = new MetaPixelEvent($event_name, $event_data, $user_data);

    // Add entity cache dependencies from context.
    foreach ($context as $dependency) {
      if ($dependency instanceof CacheableDependencyInterface) {
        $event->addCacheableDependency($dependency);
      }
    }

    // If using session storage, add session cache context.
    if ($use_session) {
      $event->addCacheContexts(['session']);
    }

    // Queue for CAPI if enabled.
    if ($capi_enabled) {
      $this->addCapiEvent($event);
    }

    // Queue for browser pixel if enabled.
    if ($browser_enabled) {
      if ($use_session) {
        $this->addDelayedBrowserEvent($event);
      }
      else {
        $this->addAnonymousBrowserEvent($event);
      }
    }

    return $event;
  }

  /**
   * {@inheritdoc}
   */
  public function getBrowserEvents(): array {
    // Get events from static storage.
    $static_events = &drupal_static('meta_pixel_events_js', []);
    $events = $static_events;
    $static_events = [];

    // Get events from session storage if session is started.
    if ($this->session->isStarted()) {
      $delayed_events = $this->session->get('meta_pixel_events_js', []);
      if ($delayed_events !== []) {
        $this->session->set('meta_pixel_events_js', []);
        $events = array_merge($delayed_events, $events);
      }
    }

    /**
     * @var \Drupal\meta_pixel\MetaPixelEvent $event
     */
    foreach ($events as $id => $event) {
      // Remove properties that shouldn't be sent to fbq().
      // @todo Make this configurable via settings.
      $data = $event->getEventData();
      unset($data['event_time']);
      unset($data['event_source_url']);
      unset($data['event_name']);
      unset($data['event_id']);
      $event->setEventData($data);
    }

    return $this->deduplicateEvents($events);
  }

  /**
   * {@inheritdoc}
   */
  public function getCapiEvents(): array {
    $events = &drupal_static('meta_pixel_events_capi', []);
    $static_events = $events;
    // Clear static storage.
    $events = [];

    return $this->deduplicateEvents($static_events);
  }

  /**
   * Deduplicates events by event ID.
   *
   * Ensures that if multiple identical events are queued (same event_id),
   * only one is kept. This can happen when events are added from multiple
   * code paths.
   *
   * @param array $events
   *   Array of MetaPixelEvent objects.
   *
   * @return array
   *   Deduplicated array of MetaPixelEvent objects keyed by event_id.
   */
  protected function deduplicateEvents(array $events): array {
    $unique_events = [];
    foreach ($events as $event) {
      $event_id = $event->getEventId();
      if (!isset($unique_events[$event_id])) {
        $unique_events[$event_id] = $event;
      }
    }
    return $unique_events;
  }

  /**
   * {@inheritdoc}
   */
  public function prepareCapiEvent(MetaPixelEvent $event): Event|null {
    // Build event data for CAPI.
    $event_data = [
      'event_name' => $event->getName(),
      'event_id' => $event->getEventId(),
      'event_time' => $event->getEventTime(),
    ];

    $custom_data = $event->getEventData();

    // Extract custom event_source_url if set.
    if (!empty($custom_data['event_source_url'])) {
      $event_data['event_source_url'] = $custom_data['event_source_url'];
      unset($custom_data['event_source_url']);
    }

    // Prepare user data for CAPI format.
    $user_data = [];
    if ($user_data_raw = $event->getUserData()) {
      $user_data = $this->prepareCapiUserData($user_data_raw);
    }

    // Build custom data with Content objects.
    $custom_data = $this->buildCustomData($custom_data);

    return $this->metaCapiClient->buildEvent($event_data, $user_data, $custom_data);
  }

  /**
   * {@inheritdoc}
   */
  public function buildUserData(array $event_data = [], array &$context = []): array {
    // Initialize structure.
    $user_data = [];

    // Determine which user account to use for account-level data.
    $user = NULL;
    $customer = NULL;
    if (!empty($context['order'])) {
      $customer = $context['order']->getCustomer();
      if ($customer && $customer->isAuthenticated()) {
        $user = $customer;
      }
    }

    // Fall back to current user if no authenticated customer from order.
    if (!$user && $id = $this->currentUser->id()) {
      /** @var \Drupal\user\UserInterface $user */
      $user = $this->entityTypeManager->getStorage('user')->load($id);
    }

    // Collect from user account if authenticated.
    if ($user && $user->isAuthenticated()) {
      // Add Drupal user ID as external ID.
      $user_data['external_id'][] = 'u_' . $user->id();
      if ($email = $user->getEmail()) {
        $user_data['email'][] = $email;
      }
    }

    // Collect from order if available.
    if (!empty($context['order'])) {
      $order = $context['order'];
      if ($email = $order->getEmail()) {
        $user_data['email'][] = $email;
      }
      if ($customer instanceof UserInterface) {
        $user_data['email'][] = $customer->getEmail();
      }
    }

    // Update context with loaded entities for alter hook implementations.
    if (!empty($user)) {
      $context['user'] = $user;
    }

    return $user_data;
  }

  /**
   * Maps singular user field names to their plural equivalents for Meta CAPI.
   *
   * Meta's Conversions API requires plural keys when sending array values
   * (e.g., 'emails' instead of 'email' when sending multiple email addresses).
   * This map enables automatic conversion from singular to plural form.
   *
   * @var array
   */
  protected static array $pluralUserFieldMap = [
    'email' => 'emails',
    'phone' => 'phones',
    'external_id' => 'external_ids',
    'first_name' => 'first_names',
    'last_name' => 'last_names',
    'city' => 'cities',
    'state' => 'states',
    'zip_code' => 'zip_codes',
    'country_code' => 'country_codes',
    'gender' => 'genders',
    'date_of_birth' => 'dates_of_birth',
  ];

  /**
   * Flattens identity sets into arrays for CAPI.
   *
   * Extracts all unique values from all identity sources (account, billing,
   * shipping profiles) and converts to the format expected by Meta
   * Conversions API. Uses singular keys for single values and plural keys
   * for arrays, as required by Meta's API.
   *
   * @param array $user_data
   *   User data with identities keyed by source.
   *
   * @return array
   *   Flattened user data ready for CAPI with proper singular/plural keys.
   */
  public function prepareCapiUserData(array $user_data): array {
    // Step 1: Flatten everything into arrays keyed by field name.
    $flattened = [];

    // Non-profile fields first.
    foreach ($user_data as $key => $value) {
      if ($key === 'profiles') {
        continue;
      }
      if (!empty($value)) {
        // Merge with existing (cast to array).
        $flattened[$key] = array_merge($flattened[$key] ?? [], (array) $value);
      }
    }

    // Flatten profiles - combines with any top-level fields.
    foreach ($user_data['profiles'] ?? [] as $profile) {
      foreach ($profile as $field => $value) {
        if (!empty($value)) {
          $flattened[$field][] = $value;
        }
      }
    }

    // Step 2: Convert to SDK format.
    $result = [];

    foreach ($flattened as $key => $values) {
      if (empty($values)) {
        unset($flattened[$key]);
        continue;
      }

      if (is_string($values)) {
        $result[$key] = $values;
        continue;
      }

      $values = array_values(array_unique(array_filter($values)));

      if (empty($values)) {
        unset($flattened[$key]);
        continue;
      }

      if (count($values) === 1) {
        $result[$key] = reset($values);
      }
      elseif (isset(self::$pluralUserFieldMap[$key])) {
        $result[self::$pluralUserFieldMap[$key]] = $values;
      }
      else {
        $result[$key] = $values;
      }
    }

    return $result;
  }

  /**
   * Extracts single user data values for browser pixel.
   *
   * Unlike CAPI which accepts arrays, the browser pixel only allows a single
   * value for each user data field. This method selects one value from each
   * available source, prioritizing complete identity sets.
   *
   * Site admins can reorder profile priority using
   * hook_meta_pixel_event_data_alter().
   *
   * @param array $user_data
   *   User data with identities keyed by source.
   *
   * @return array
   *   Single values mapped to browser pixel format with abbreviated keys
   *   ('em' for email, 'fn' for first name, 'ln' for last name, etc.).
   *
   * @todo This method is not currently used and may not be needed. CAPI
   *   events are more reliable for user data transmission. Additionally,
   *   adding PII as JavaScript variables may pose privacy concerns. Current
   *   best practices rely on CAPI for user data transmission. Consider
   *   removing in a future version if it remains unused.
   */
  public function prepareBrowserUserData(array $user_data): array {
    $browser_data = [];

    // External ID - pick first.
    if (!empty($user_data['external_ids'])) {
      $browser_data['external_id'] = reset($user_data['external_ids']);
    }

    // Email - pick first.
    if (!empty($user_data['emails'])) {
      $browser_data['em'] = reset($user_data['emails']);
    }

    // Phone - pick first.
    if (!empty($user_data['phones'])) {
      $browser_data['ph'] = reset($user_data['phones']);
    }

    // Identity set - use first profile.
    $profile = NULL;
    if (!empty($user_data['profiles'])) {
      $profile = reset($user_data['profiles']);
    }

    // Map identity fields to browser pixel format.
    if ($profile) {
      if (!empty($profile['first_name'])) {
        $browser_data['fn'] = strtolower(trim($profile['first_name']));
      }
      if (!empty($profile['last_name'])) {
        $browser_data['ln'] = strtolower(trim($profile['last_name']));
      }
      if (!empty($profile['city'])) {
        $browser_data['ct'] = strtolower(trim($profile['city']));
      }
      if (!empty($profile['state'])) {
        $browser_data['st'] = strtolower(trim($profile['state']));
      }
      if (!empty($profile['zip_code'])) {
        $browser_data['zp'] = trim($profile['zip_code']);
      }
      if (!empty($profile['country_code'])) {
        $browser_data['country'] = strtolower(trim($profile['country_code']));
      }
    }

    return array_filter($browser_data);
  }

  /**
   * Builds custom data for CAPI from event data.
   *
   * Normalizes event parameters and converts 'contents' arrays to Facebook
   * SDK Content objects required by Meta's Conversions API. Handles both
   * array-based content data and pre-constructed Content objects.
   *
   * @param array $event_data
   *   Event data containing parameters like contents, value, currency.
   *
   * @return array
   *   Custom data array formatted for Meta CAPI, with contents converted
   *   to Content objects if present.
   */
  protected function buildCustomData(array $event_data): array {
    $custom_data = $event_data;

    // Handle contents array - convert to Facebook SDK Content objects for CAPI.
    if (isset($event_data['contents']) && is_array($event_data['contents'])) {
      $contents = [];
      foreach ($event_data['contents'] as $content_data) {
        if (is_array($content_data)) {
          // Convert array to Content object with all supported fields.
          $content = new Content();

          if (isset($content_data['id'])) {
            $content->setProductId($content_data['id']);
          }
          if (isset($content_data['quantity'])) {
            $content->setQuantity((int) $content_data['quantity']);
          }
          if (isset($content_data['item_price'])) {
            $content->setItemPrice((float) $content_data['item_price']);
          }
          if (isset($content_data['brand'])) {
            $content->setBrand($content_data['brand']);
          }
          if (isset($content_data['category'])) {
            $content->setCategory($content_data['category']);
          }
          if (isset($content_data['title'])) {
            $content->setTitle($content_data['title']);
          }
          if (isset($content_data['delivery_category'])) {
            $content->setDeliveryCategory($content_data['delivery_category']);
          }

          $contents[] = $content;
        }
        elseif ($content_data instanceof Content) {
          // Already a Content object - use as-is.
          $contents[] = $content_data;
        }
      }
      $custom_data['contents'] = $contents;
    }

    return $custom_data;
  }

  /**
   * Queues an event for CAPI transmission.
   *
   * CAPI events are ALWAYS stored in static array only, never session,
   * because they're sent during kernel.terminate of the same request.
   * This ensures events can be transmitted even when session is not
   * available or has been written.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The event object to queue for CAPI.
   */
  protected function addCapiEvent(MetaPixelEvent $event): void {
    $events = &drupal_static('meta_pixel_events_capi', []);
    $events[] = $event;
  }

  /**
   * Queues an event for immediate browser pixel tracking.
   *
   * Events stored here are rendered in the same request via either
   * hook_page_attachments (for HTML responses) or ResponseSubscriber
   * (for AJAX responses). Uses static storage for efficiency.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The event object to queue for browser tracking.
   */
  protected function addAnonymousBrowserEvent(MetaPixelEvent $event): void {
    $events = &drupal_static('meta_pixel_events_js', []);
    $events[] = $event;
  }

  /**
   * Queues an event in session storage for delayed browser tracking.
   *
   * Used for POST-redirect-GET patterns where the event must survive a
   * redirect (e.g., Purchase event after checkout form submission that
   * redirects to a thank-you page). The event will be retrieved and
   * rendered on the next page load.
   *
   * @param \Drupal\meta_pixel\MetaPixelEvent $event
   *   The event object to store in session.
   *
   * @throws \Exception
   *   When session storage fails.
   */
  protected function addDelayedBrowserEvent(MetaPixelEvent $event): void {
    try {
      $delayed_events = $this->session->get('meta_pixel_events_js', []);
      $delayed_events[] = $event;
      $this->session->set('meta_pixel_events_js', $delayed_events);
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to store Meta Pixel event in session: @message', [
        '@message' => $e->getMessage(),
      ]);
    }
  }

  /**
   * {@inheritDoc}
   */
  public function isCapiEnabled(string $plugin_id): bool {
    $config = $this->configFactory->get('meta_pixel.settings');
    $has_pixel_id = !empty($config->get('pixel_id'));
    $has_access_token = !empty($config->get('capi.access_token'));
    return $has_pixel_id && $has_access_token && $this->configFactory->get('meta_pixel.events')->get('capi_enabled.' . $plugin_id);
  }

  /**
   * {@inheritDoc}
   */
  public function isBrowserEnabled(string $plugin_id): bool {
    $has_pixel_id = !empty($this->configFactory->get('meta_pixel.settings')->get('pixel_id'));
    return $has_pixel_id && $this->configFactory->get('meta_pixel.events')->get('browser_enabled.' . $plugin_id);
  }

  /**
   * {@inheritDoc}
   */
  public function isAnyEnabled(string $plugin_id): bool {
    return ($this->isBrowserEnabled($plugin_id) || $this->isCapiEnabled($plugin_id));
  }

}
