<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\meta_pixel\EventCollector;
use Drupal\meta_pixel\MetaCapiClient;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\TerminateEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Adds all Meta Pixel CAPI events collected in one api call.
 */
final class TerminateSubscriber implements EventSubscriberInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  private EventCollector $eventCollector;

  /**
   * The Meta CAPI Client.
   *
   * @var \Drupal\meta_pixel\MetaCapiClient
   */
  private MetaCapiClient $metaCapiClient;

  /**
   * The config settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  private ImmutableConfig $config;

  /**
   * Constructs a TerminateSubscriber object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector.
   * @param \Drupal\meta_pixel\MetaCapiClient $metaCapiClient
   *   The Meta CAPI Client.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(EventCollector $eventCollector, MetaCapiClient $metaCapiClient, ConfigFactoryInterface $config_factory) {
    $this->eventCollector = $eventCollector;
    $this->metaCapiClient = $metaCapiClient;
    $this->config = $config_factory->get('meta_pixel.settings');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::TERMINATE => ['onTerminate', 10],
    ];
  }

  /**
   * Makes a CAPI request from the collected events.
   *
   * @param \Symfony\Component\HttpKernel\Event\TerminateEvent $event
   *   The terminate event.
   */
  public function onTerminate(TerminateEvent $event): void {
    // Only act on main request.
    if (!$event->isMainRequest()) {
      return;
    }
    // If (!\Drupal::request()->isMethodSafe()) {
    // return;
    // }

    // Get all queued events.
    $events = $this->eventCollector->getCapiEvents();
    $capi_events = [];
    foreach ($events as $pixel_event) {
      if ($capi_event = $this->eventCollector->prepareCapiEvent($pixel_event)) {
        $capi_events[] = $capi_event;
      }
    }
    if ($capi_events) {
      if ($this->config->get('capi.enable_logging')) {
        \Drupal::logger('meta_pixel')->info('Send CAPI events:<br/><pre>@events</pre>', [
          '@events' => print_r($capi_events, TRUE),
        ]);
      }
      $this->metaCapiClient->sendRequest($capi_events);
    }
  }

}
