<?php

declare(strict_types=1);

namespace Drupal\meta_pixel\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure which Meta Pixel events are tracked and where.
 *
 * Uses vertical tabs to organize per-event configuration including
 * browser pixel tracking, CAPI tracking, and optional plugin-specific fields.
 */
class EventsForm extends ConfigFormBase {

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * The Meta Pixel event plugin manager.
   *
   * @var \Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager
   */
  protected MetaPixelEventPluginManager $eventPluginManager;

  /**
   * Constructs an EventsForm object.
   *
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\meta_pixel\Plugin\MetaPixelEventPluginManager $event_plugin_manager
   *   The event plugin manager.
   */
  public function __construct(
    ModuleHandlerInterface $module_handler,
    MetaPixelEventPluginManager $event_plugin_manager
  ) {
    $this->moduleHandler = $module_handler;
    $this->eventPluginManager = $event_plugin_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('module_handler'),
      $container->get('meta_pixel.event_plugin_manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames() {
    return ['meta_pixel.events'];
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'meta_pixel_events_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state) {
    $config = $this->config('meta_pixel.events');

    // Get all plugin definitions from the plugin manager.
    $plugin_definitions = $this->eventPluginManager->getDefinitions();

    if (empty($plugin_definitions)) {
      $form['no_events'] = [
        '#type' => 'markup',
        '#markup' => '<p>' . $this->t('No event plugins are currently available.') . '</p>',
      ];
      return parent::buildForm($form, $form_state);
    }

    // Sort plugins by label for consistent display.
    uasort($plugin_definitions, function ($a, $b) {
      return strcmp((string) $a['label'], (string) $b['label']);
    });


    $form['description'] = [
      '#type' => 'markup',
      '#markup' => '<p>' . $this->t('Configure Meta Pixel event tracking. Each event can be enabled independently for browser pixel (JavaScript) and server-side CAPI tracking.') . '</p>',
    ];


    // Create vertical tabs container.
    $form['events'] = [
      '#type' => 'vertical_tabs',
      '#title' => $this->t('Events'),
    ];

    // Get stored configuration.
    $browser_enabled = $config->get('browser_enabled') ?? [];
    $capi_enabled = $config->get('capi_enabled') ?? [];

    // Build a tab for each plugin.
    foreach ($plugin_definitions as $plugin_id => $plugin_definition) {
      $plugin_key = str_replace([':', '-'], '_', $plugin_id);

      $event_config = $events[$plugin_id] ?? [];
      $event_plugin = $this->eventPluginManager->createInstance($plugin_id, $event_config);

      // Create a details element for each plugin (becomes a tab).
      $form['plugin_' . $plugin_key] = [
        '#type' => 'details',
        '#title' => $plugin_definition['label'],
        '#group' => 'events',
        '#tree' => TRUE,
      ];

      // Event name and description.
      $form['plugin_' . $plugin_key]['info'] = [
        '#type' => 'item',
        '#title' => $plugin_definition['event_name'],
        '#markup' => '<p>' . $plugin_definition['description'] . '</p>',
      ];


      // Tracking options section.
      $form['plugin_' . $plugin_key]['tracking'] = [
        '#type' => 'fieldset',
        '#title' => $this->t('Tracking Options'),
      ];

      $form['plugin_' . $plugin_key]['tracking']['browser_enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable browser pixel tracking'),
        '#description' => $this->t('Track this event via JavaScript (fbq). Works for all users but may be blocked by ad blockers or browser privacy features.'),
        '#default_value' => $browser_enabled[$plugin_id] ?? FALSE,
      ];

      $form['plugin_' . $plugin_key]['tracking']['capi_enabled'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable server-side tracking (CAPI)'),
        '#description' => $this->t('Send this event from your server to Meta. Bypasses browser blockers and works for iOS/Safari users.'),
        '#default_value' => $capi_enabled[$plugin_id] ?? FALSE,
      ];

      // Placeholder for future plugin-specific config fields.
      // Plugins can implement a buildConfigurationForm() method to add fields here.
      // Example:
      // $plugin = $this->eventPluginManager->createInstance($plugin_id);
      // if (method_exists($plugin, 'buildConfigurationForm')) {
      //   $form['plugin_' . $plugin_key]['plugin_config'] = [
      //     '#type' => 'details',
      //     '#title' => $this->t('Advanced Configuration'),
      //     '#open' => FALSE,
      //   ];
      //   $plugin_form = $plugin->buildConfigurationForm([], $form_state);
      //   $form['plugin_' . $plugin_key]['plugin_config'] += $plugin_form;
      // }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    $config = $this->config('meta_pixel.events');
    $plugin_definitions = $this->eventPluginManager->getDefinitions();

    $browser_enabled = [];
    $capi_enabled = [];

    foreach ($plugin_definitions as $plugin_id => $plugin_definition) {
      $plugin_key = str_replace([':', '-'], '_', $plugin_id);
      $plugin_values = $form_state->getValue('plugin_' . $plugin_key);

      if ($plugin_values && isset($plugin_values['tracking'])) {
        $browser_enabled[$plugin_id] = !empty($plugin_values['tracking']['browser_enabled']);
        $capi_enabled[$plugin_id] = !empty($plugin_values['tracking']['capi_enabled']);

        // Future: Handle plugin-specific configuration.
        // if (isset($plugin_values['plugin_config'])) {
        //   $plugin = $this->eventPluginManager->createInstance($plugin_id);
        //   if (method_exists($plugin, 'submitConfigurationForm')) {
        //     $plugin->submitConfigurationForm($plugin_values['plugin_config'], $form_state);
        //   }
        // }
      }
    }

    $config
      ->set('browser_enabled', $browser_enabled)
      ->set('capi_enabled', $capi_enabled)
      ->save();

    $this->messenger()->addStatus($this->t('Meta Pixel event configuration has been saved.'));

    parent::submitForm($form, $form_state);
  }

}
