<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Core\Cache\Cache;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\Security\TrustedCallbackInterface;
use Psr\Log\LoggerInterface;
use Symfony\Component\Yaml\Yaml;

/**
 * Lazy builder for ViewContent event tracking.
 */
class LazyBuilderEmitter implements TrustedCallbackInterface {

  /**
   * The event collector.
   *
   * @var \Drupal\meta_pixel\EventCollector
   */
  private EventCollector $eventCollector;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  private EntityTypeManagerInterface $entityTypeManager;

  /**
   * The logger channel.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected RouteMatchInterface $routeMatch;

  /**
   * Constructs a ViewContentLazyBuilder object.
   *
   * @param \Drupal\meta_pixel\EventCollector $eventCollector
   *   The event collector.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    EventCollector $eventCollector,
    EntityTypeManagerInterface $entityTypeManager,
    LoggerInterface $logger,
    RouteMatchInterface $route_match,
  ) {
    $this->eventCollector = $eventCollector;
    $this->entityTypeManager = $entityTypeManager;
    $this->logger = $logger;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function trustedCallbacks() {
    return ['emitEvent'];
  }
  /**
   *
   */
  public function emitEvent(string $plugin_id, string $event_name = '', string $custom_properties = '', string $debug = '0'): array {
    // Check if tracking is enabled for this event.
    if (!$this->eventCollector->isAnyEnabled($plugin_id)) {
      // Return cached "disabled" state that invalidates when config changes.
      return [
        '#cache' => [
          'max-age' => Cache::PERMANENT,
          'tags' => [
            'config:meta_pixel.settings',
            'config:meta_pixel.events',
          ],
        ],
      ];
    }

    $build = [
      '#cache' => [
        'max-age' => 0,
      ],
    ];

    // Build event data array.
    // Extract entities from the current route.
    $data = $this->extractEntitiesFromRoute();

    // Add custom event name if specified.
    if (!empty($event_name)) {
      $data['event_name'] = $event_name;
    }

    // Parse custom properties if provided.
    if (!empty($custom_properties)) {
      try {
        $custom_props = Yaml::parse($custom_properties);
        if (is_array($custom_props)) {
          $data = array_merge($data, $custom_props);
        }
      }
      catch (\Exception $e) {
        // Log error but don't break page rendering.
        $this->logger->error('Failed to parse custom properties in Meta Pixel Emitter: @error', [
          '@error' => $e->getMessage(),
        ]);
      }
    }

    // Add the event to the collector.
    $this->eventCollector->addEvent($plugin_id, $data);

    // Get collected events and attach to drupalSettings.
    $events = [];
    foreach ($this->eventCollector->getBrowserEvents() as $event) {
      $events[] = [
        'event_name' => $event->getName(),
        'event_data' => $event->getEventData(),
        'event_id' => $event->getEventId(),
      ];
    }

    if ($events) {
      $build['#attached']['drupalSettings']['metaPixel']['events'] = $events;
      $debug = (bool) $debug;
      if ($debug) {
        $yaml = Yaml::dump($events, 4, 2);
        $build['debug'] = [
          '#type' => 'item',
          '#title' => t('Event Data'),
          '#markup' => '<pre>' . $yaml . '</pre>',
        ];

        $user_data_raw = $event->getUserData();
        if ($user_data_raw) {
          $user_data = $this->eventCollector->prepareCapiUserData($user_data_raw);
          $yaml_user = Yaml::dump($user_data, 4, 2);
          $build['debug_user'] = [
            '#type' => 'item',
            '#title' => t('User Data (only sent to CAPI)'),
            '#markup' => '<pre>' . $yaml_user . '</pre>',
          ];
        }
        //['#markup'] = '<pre>' . $yaml_user . '</pre>';
      }
    }

    return $build;
  }

  /**
   * Extracts all entities from the current route parameters.
   *
   * This automatically finds any entity objects in the route (node, user,
   * commerce_product, etc.) and adds them to the data array using their
   * entity type as the key.
   *
   * @return array
   *   Array of entities keyed by entity type.
   */
  protected function extractEntitiesFromRoute(): array {
    $data = [];

    // Get all route parameters.
    $parameters = $this->routeMatch->getParameters();

    // Extract entity objects.
    foreach ($parameters as $key => $parameter) {
      if ($parameter instanceof EntityInterface) {
        // Use entity type as key (e.g., 'node', 'commerce_product').
        $entity_type = $parameter->getEntityTypeId();
        $data[$entity_type] = $parameter;

        // Also add with the original parameter name for flexibility.
        // This handles cases like 'product' vs 'commerce_product'.
        if ($key !== $entity_type) {
          $data[$key] = $parameter;
        }
      }
    }

    return $data;
  }

}
