<?php

namespace Drupal\meta_pixel\Logger;

use Drupal\Component\Serialization\Json;
use FacebookAds\Http\RequestInterface;
use FacebookAds\Http\ResponseInterface;
use FacebookAds\Logger\LoggerInterface as FacebookLoggerInterface;
use Psr\Log\LoggerInterface;

/**
 * Implements Facebook LoggerInterface and logs to the Drupal log.
 *
 * The Facebook LoggerInterface is incompatible with Psr one in Drupal.
 */
class FacebookLogger implements FacebookLoggerInterface {

  /**
   * Drupal logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected $logger;

  public function __construct(LoggerInterface $logger) {
    $this->logger = $logger;
  }

  /**
   * {@inheritdoc}
   */
  public function log($level, $message, array $context = []) {
    $this->logger->log($level, $message, $context);
  }

  /**
   * {@inheritdoc}
   */
  public function logRequest($level, RequestInterface $request, array $context = []) {
    $query_params = json_encode($request->getQueryParams(), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
    $body_params = json_encode($request->getBodyParams(), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    $this->log($level, 'Meta API request: method @method<br><strong>Query params:</strong><pre>@query_params</pre><strong>Body params:</strong><pre>@body_params</pre>', [
      '@method' => $request->getMethod(),
      '@query_params' => $query_params,
      '@body_params' => $body_params,
    ]);
  }

  /**
   * {@inheritdoc}
   */
  public function logResponse($level, ResponseInterface $response, array $context = []) {
    $body = json_encode(json_decode($response->getBody()), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

    $this->log($level, 'Meta API response: @status_code<br><strong>Body:</strong><pre>@body</pre>', [
      '@status_code' => $response->getStatusCode(),
      '@body' => $body,
    ]);
  }
}
