<?php

declare(strict_types=1);

namespace Drupal\meta_pixel;

use Drupal\Core\Config\ConfigFactoryInterface;
use FacebookAds\ParamBuilder;
use Psr\Log\LoggerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Factory service for creating Meta Parameter Builder instances.
 *
 * Wraps Meta's ParamBuilder library with Drupal-specific configuration
 * and error handling. Provides a single point of configuration for domain
 * management and cookie handling.
 */
class MetaParamBuilderFactory {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The configuration factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected ConfigFactoryInterface $configFactory;

  /**
   * The logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Cached ParamBuilder instance.
   *
   * @var \FacebookAds\ParamBuilder|null
   */
  protected ?ParamBuilder $paramBuilder = NULL;

  /**
   * Constructs a MetaParamBuilderFactory.
   *
   * @param \Symfony\Component\HttpFoundation\RequestStack $request_stack
   *   The request stack.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The configuration factory.
   * @param \Psr\Log\LoggerInterface $logger
   *   The logger channel.
   */
  public function __construct(
    RequestStack $request_stack,
    ConfigFactoryInterface $config_factory,
    LoggerInterface $logger,
  ) {
    $this->requestStack = $request_stack;
    $this->configFactory = $config_factory;
    $this->logger = $logger;
  }

  /**
   * Gets or creates a ParamBuilder instance.
   *
   * @return \FacebookAds\ParamBuilder|null
   *   The parameter builder instance, or NULL if initialization fails.
   */
  public function getParamBuilder(): ?ParamBuilder {
    if ($this->paramBuilder !== NULL) {
      return $this->paramBuilder;
    }

    try {
      $domains = $this->getDomains();

      if (empty($domains)) {
        $this->logger->error('Parameter Builder enabled but no domains configured. Configure domains at /admin/config/services/meta-pixel');
        return NULL;
      }

      $this->paramBuilder = new ParamBuilder($domains);

      return $this->paramBuilder;
    }
    catch (\Exception $e) {
      $this->logger->error('Failed to initialize Meta Parameter Builder: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Processes the current HTTP request through Parameter Builder.
   *
   * Extracts Facebook click IDs (fbc), browser IDs (fbp), and client IP
   * addresses from the request. Returns updated cookies that should be set.
   *
   * @return array
   *   Array of cookie settings returned by Parameter Builder, or empty array
   *   on failure. Each element is a CookieSetting object with properties:
   *   - name: Cookie name (_fbc, _fbp)
   *   - value: Cookie value
   *   - domain: Cookie domain
   *   - maxAge: Cookie expiration time in seconds
   */
  public function processRequest(): array {
    $paramBuilder = $this->getParamBuilder();
    if (!$paramBuilder) {
      return [];
    }

    $request = $this->requestStack->getCurrentRequest();
    if (!$request) {
      return [];
    }

    try {
      // Extract request data following Meta's documentation.
      $host = $request->getHost();
      $query_params = $request->query->all();
      $cookies = $request->cookies->all();
      $referer = $request->headers->get('referer');
      $x_forwarded_for = $request->headers->get('X-Forwarded-For');
      $remote_address = $request->getClientIp();

      // Process request through Parameter Builder.
      // Returns array of CookieSetting objects.
      $updated_cookies = $paramBuilder->processRequest(
        $host,
        $query_params,
        $cookies,
        $referer,
        $x_forwarded_for,
        $remote_address
      );

      return $updated_cookies ?? [];
    }
    catch (\Exception $e) {
      $this->logger->error('Parameter Builder processRequest failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return [];
    }
  }

  /**
   * Gets the Facebook Click ID from processed request.
   *
   * Must call processRequest() first.
   *
   * @return string|null
   *   The fbc value, or NULL if not available.
   */
  public function getFbc(): ?string {
    $paramBuilder = $this->getParamBuilder();
    if (!$paramBuilder) {
      return NULL;
    }

    try {
      return $paramBuilder->getFbc();
    }
    catch (\Exception $e) {
      $this->logger->error('Parameter Builder getFbc failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets the Facebook Browser ID from processed request.
   *
   * Must call processRequest() first.
   *
   * @return string|null
   *   The fbp value, or NULL if not available.
   */
  public function getFbp(): ?string {
    $paramBuilder = $this->getParamBuilder();
    if (!$paramBuilder) {
      return NULL;
    }

    try {
      return $paramBuilder->getFbp();
    }
    catch (\Exception $e) {
      $this->logger->error('Parameter Builder getFbp failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets the client IP address from processed request.
   *
   * Parameter Builder automatically detects IPv6 when available and falls
   * back to IPv4. Must call processRequest() first.
   *
   * @return string|null
   *   The client IP address (IPv6 preferred, IPv4 fallback), or NULL if not
   *   available.
   */
  public function getClientIpAddress(): ?string {
    $paramBuilder = $this->getParamBuilder();
    if (!$paramBuilder) {
      return NULL;
    }

    try {
      return $paramBuilder->getClientIpAddress();
    }
    catch (\Exception $e) {
      $this->logger->error('Parameter Builder getClientIpAddress failed: @message', [
        '@message' => $e->getMessage(),
      ]);
      return NULL;
    }
  }

  /**
   * Gets configured domains for Parameter Builder.
   *
   * Extracts domain list from module configuration. These are the top-level
   * domains (eTLD+1) where Meta Pixel cookies will be set.
   *
   * @return array
   *   Array of domain strings (e.g., ['example.com', 'test.com']).
   */
  protected function getDomains(): array {
    $config = $this->configFactory->get('meta_pixel.settings');
    $domains_string = $config->get('param_builder.domains');

    if (empty($domains_string)) {
      return [];
    }

    // Split by comma and trim whitespace.
    $domains = array_map('trim', explode(',', $domains_string));

    // Remove empty values.
    return array_filter($domains);
  }

}
